      SUBROUTINE CALC
C***********************************************************************
C                 CALC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Controls Flow and Processing of CALCulation Modules
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To move check for AREA source first for optimization.
C                    R. W. Brode, PES - 5/12/99
C
C        MODIFIED:   To add call for new source type of OPENPIT.
C                    R. W. Brode, PES - 9/30/94
C
C        INPUTS:  Arrays of Source Parameters
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Array of 1-hr CONC or DEPOS Values for Each Source/Receptor
C
C        CALLED FROM:   HRLOOP
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'CALC'
      PATH = 'CN'

C     Begin Source LOOP
      DO ISRC = 1, NUMSRC
         IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
C           Calculate Area Source Values for Rectangles  ---   CALL ACALC
            CALL ACALC
         ELSE IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
C           Calculate Point Source Values                ---   CALL PCALC
            CALL PCALC
         ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
C           Calculate Volume Source Values               ---   CALL VCALC
            CALL VCALC
         ELSE IF (SRCTYP(ISRC) .EQ. 'AREAPOLY') THEN
C           Calculate Area Source Values for Polygons    ---   CALL ACALC
            CALL ACALC
         ELSE IF (SRCTYP(ISRC) .EQ. 'AREACIRC') THEN
C           Calculate Area Source Values for Circles     ---   CALL ACALC
            CALL ACALC
         ELSE IF (SRCTYP(ISRC) .EQ. 'OPENPIT') THEN
C           Calculate OpenPit Source Values              ---   CALL OCALC
            CALL OCALC            
         END IF
      END DO
C     End Source LOOP

      RETURN
      END

      SUBROUTINE PCALC
C***********************************************************************
C                 PCALC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates concentration or deposition values
C                 for POINT sources
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        MODIFIED BY D. Strimaitis, SRC (for Wet & Dry DEPOSITION)
C        MODIFIED BY D. Strimaitis, SRC (for COMPLEX I -Intermediate
C                                        Terrain Processing)
C
C        DATE:    December 15, 1993
C
C        MODIFIED:   To skip receptor if it is more than 80km from source
C                    for TOXICS option.
C                    R. W. Brode, PES - 02/19/99
C
C        MODIFIED:   To allow use with EVENT processing.
C                    R. W. Brode, PES - 12/2/98
C
C        MODIFIED:   To add call for new source type of OPENPIT.
C                    R. W. Brode, PES - 9/30/94
C
C        MODIFIED BY D. Strimaitis, SRC (for Dry DEPOSITION)
C        (DATE:    February 15, 1993)
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: 1-hr CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   CALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I

C     Variable Initializations
      MODNAM = 'PCALC'

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFACT
      CALL EMFACT(QS)

C     Set Deposition Variables for this Source
      IF (LDPART .OR. LWPART .OR. LDGAS) THEN
C        Calculate Deposition Velocities for this Source    ---   CALL VDP
         CALL VDP
      END IF

C     Calculate Scavenging Ratios for this Source           ---   CALL SCAVRAT
      IF (LWPART .OR. LWGAS) CALL SCAVRAT

      IF ((QTK.NE.0.0) .AND. (STABLE .OR. (HS.LE.ZI) .OR.
     &                        DEPOS .OR. WDEP)) THEN
C        Calculate Buoyancy and Momentum Fluxes
         IF (TS .LT. TA)  TS = TA
         FB = (0.25/TS)*(VS*DS*DS)*G*(TS-TA)
         FM = (0.25/TS)*(VS*DS*DS)*VS*TA
C        Adjust Wind Speed to Stack Height                  ---   CALL WSADJ
         CALL WSADJ
C        Calculate Distance to Final Rise                   ---   CALL DISTF
         CALL DISTF
C        Set Wake and Building Type Switches                ---   CALL WAKFLG
         CALL WAKFLG
C        Initialize FSTREC Logical Switch for First Receptor of Loop
         FSTREC = .TRUE.
C        Initialize FSTREC Logical Switch for First CMP1 Receptor of Loop
         FSTCMP = .TRUE.
         IF (LDPART .OR. LWPART) THEN
C           Calculate Min Sigma-z for Settled Plume @ Surface --- CALL SETSZMN
            CALL SETSZMN
         END IF

C        Begin Receptor LOOP
         DO IREC = 1, NUMREC
C           Calculate Down and Crosswind Distances          ---   CALL XYDIST
            IF (EVONLY) THEN
               CALL XYDIST(IEVENT)
            ELSE
               CALL XYDIST(IREC)
            END IF
            IF (ABS(Y) .GT. 1.191754*X) THEN
C              Receptor is at least 50 deg. off the plume centerline
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            ELSE IF (DISTR .LT. 0.99 .OR. DISTR .LT. 3.*ZLB) THEN
C              Receptor Too Close to Source for Calculation
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            ELSE IF (TOXICS .AND. DISTR .GT. 80000.) THEN
C              Receptor is beyond 80km from source.
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            ELSE

CRWB           Modifications to Integrate COMPLEX1 Algorithms.
CRWB           Get Plume Heights and Check for INTERMEDIATE TERRAIN Regime
               IF (NOCMPL) THEN
                  CALL PHEFF(X,DHP,HEFLAT)
                  CALL STERAD(HEFLAT,ZELEV,HE)
                  SIMPLE = .TRUE.
                  INTERM = .FALSE.
                  COMPLX = .FALSE.
C                 Set HECOMP = HEFLAT for later check versus ZI
                  HECOMP = HEFLAT
               ELSE IF (NOSMPL) THEN
C                 Note: Use radial distance, DISTR, for COMPLEX1 plume height.
                  CALL PHEFFC(DISTR,DHPCMP,HECOMP)
                  CALL CTERAD(HECOMP,ZELEV,HECMP1,CORR)
                  COMPLX = .TRUE.
                  INTERM = .FALSE.
                  SIMPLE = .FALSE.
C                 Set HEFLAT = HECOMP for later check versus ZI
                  HEFLAT = HECOMP
               ELSE
                  CALL PHEFF(X,DHP,HEFLAT)
                  CALL STERAD(HEFLAT,ZELEV,HE)
                  IF (ELEV) THEN
                     CALL PHEFFC(DISTR,DHPCMP,HECOMP)
                     CALL CTERAD(HECOMP,ZELEV,HECMP1,CORR)
C                    Set the Simple/Intermediate/Complex Terrain Flags
                     CALL ITSET
                  ELSE
                     SIMPLE = .TRUE.
                     HECOMP = HEFLAT
                  END IF
               END IF

               IF (STABLE .OR. (HEFLAT.LE.ZI) .OR. (HECOMP.LE.ZI) .OR.
     &             DEPOS  .OR.  WDEP) THEN

                  IF (SIMPLE .OR. INTERM) THEN
C                    Determine Simple Terrain Sigmas        ---   CALL PDIS
                     CALL PDIS(X,SY,SZ,XY,XZ,SBID)
                  END IF
                  IF (COMPLX .OR. INTERM) THEN
C                    Determine Complex Terrain Sigmas       ---   CALL PDISC
                     CALL PDISC(DISTR,SZCMP1,XZCMP1,SBCMP1)
                  END IF

C                 Determine Deposition Correction Factors for Gases
                  IF (LWGAS) THEN
C                    Initialize wet source depletion factor to unity.
                     WQCORG = 1.
                     WQCORGC = 1.
                     IF (WDPLETE) THEN
C                       Determine source depletion factor
C                       from wet removal (GASES)
                        IF (SIMPLE .OR. INTERM) THEN
C                          Simple Terrain Model
                           WQCORG = EXP(-GSCVRT*X/US)
                        ENDIF
                        IF (COMPLX .OR. INTERM) THEN
C                          Complex Terrain Model - use radial distance
                           WQCORGC = EXP(-GSCVRT*DISTR/US)
                        ENDIF
                     ENDIF
                  ENDIF

C                 Apply Intermediate Terrain Logic
                  IF (SIMPLE) THEN
C                    Simple Terrain Model Only              ---   CALL PSIMPL
                     CALL PSIMPL
                  ELSE IF (COMPLX) THEN
C                    Complex Terrain Model Only             ---   CALL PCOMPL
                     CALL PCOMPL
                  ELSE IF (INTERM) THEN
C                    Initialize simple and complex terrain holding variables
                     SIMCON = 0.0
                     COMCON = 0.0
                     DO ITYP = 1, NUMTYP
                        SIMPL(ITYP) = 0.
                        COMPL(ITYP) = 0.
                        IF (WETSCIM) SIMPLD(ITYP) = 0.
                        IF (WETSCIM) COMPLD(ITYP) = 0.
                     END DO
C                    Determine Which Model Predicts the Larger Conc.
C                    Save Simple Terrain Conc.           ---   CALL PSIMPL
                     CALL PSIMPL
                     DO ITYP = 1, NUMTYP
                        SIMPL(ITYP) = HRVAL(ITYP)
                        IF (WETSCIM) SIMPLD(ITYP) = HRVALD(ITYP)
                     END DO
C                    Save Complex Terrain Conc.          ---   CALL PCOMPL
                     CALL PCOMPL
                     DO ITYP = 1, NUMTYP
                        COMPL(ITYP) = HRVAL(ITYP)
                        IF (WETSCIM) COMPLD(ITYP) = HRVALD(ITYP)
                     END DO
C                    Report Result for Model that Produces the Larger
C                    Concentration
                     IF (SIMCON .GE. COMCON) THEN
                        DO ITYP = 1, NUMTYP
                           HRVAL(ITYP) = SIMPL(ITYP)
                           IF (WETSCIM) HRVALD(ITYP) = SIMPLD(ITYP)
                        END DO
                     ELSE
                        DO ITYP = 1, NUMTYP
                           HRVAL(ITYP) = COMPL(ITYP)
                           IF (WETSCIM) HRVALD(ITYP) = COMPLD(ITYP)
                        END DO
                     END IF
                  END IF

C                 Sum HRVAL to AVEVAL and ANNVAL Arrays  ---   CALL SUMVAL
                  IF (EVONLY) THEN
                     CALL EV_SUMVAL
                  ELSE
                     CALL SUMVAL
                  END IF

               ELSE
                  DO ITYP = 1, NUMTYP
                     HRVAL(ITYP) = 0.0
                     IF (WETSCIM) HRVALD(ITYP) = 0.0
                  END DO
               END IF

C              Write DEBUG Information related to Terrain and Removal
               IF (DEBUG) THEN
                  WRITE(IOUNIT,*)
                  WRITE(IOUNIT,*) 'HOUR, RECEPTOR : ',IHOUR,IREC
                  WRITE(IOUNIT,*) 'PCALC: HRVAL(final) = ',HRVAL
                 IF (LDPART .OR. LWPART) THEN
                  WRITE(IOUNIT,*) 'PCALC: Particle Removal --------'
                  WRITE(IOUNIT,*) 'WQCOR  = ',(WQCOR(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'WQCORC = ',(WQCORC(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'DQCOR  = ',(DQCOR(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'DQCORC = ',(DQCORC(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'PCORZD = ',(PCORZD(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'PCORZDC= ',(PCORZDC(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'PCORZR = ',(PCORZR(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'PCORZRC= ',(PCORZRC(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'SZCOR  = ',(SZCOR(I),I=1,NPD)
                  WRITE(IOUNIT,*) 'SZCORC = ',(SZCORC(I),I=1,NPD)
                 ENDIF
                  WRITE(IOUNIT,*) 'PCALC: Gas Removal -------------'
                  WRITE(IOUNIT,*) 'WQCORG, WQCORGC = ',WQCORG,WQCORGC
                  WRITE(IOUNIT,*) 'PCALC: Concentration -----------'
                  WRITE(IOUNIT,*) 'SIMPL, COMPL    = ',SIMPL,COMPL
               END IF

            END IF
         END DO
C        End Receptor LOOP
      END IF

      RETURN
      END


      SUBROUTINE ITSET
C***********************************************************************
C                 ITSET Module of the ISC Short Term Model - Version 2
C
C        PURPOSE:    To set intermediate terrain variables, based on
C                    complex terrain plume height and terrain height.
C
C        PROGRAMMER: Roger W. Brode, PES, Inc.
C
C        DATE:       September 30, 1994
C
C        INPUTS:     HECOMP = Complex Terrain Plume Height, without
C                             Terrain Adjustment Factors (through COMMON)
C
C        OUTPUTS:    SIMPLE, COMPLX, INTERM = Intermediate terrain
C                    logical control variables (through COMMON)
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'ITSET'
      SIMPLE = .FALSE.
      INTERM = .FALSE.
      COMPLX = .FALSE.

      IF (HS .GE. HTER) THEN
         SIMPLE = .TRUE.
      ELSE IF (HECOMP .GT. HTER) THEN
         INTERM = .TRUE.
      ELSE
         COMPLX = .TRUE.
      END IF

C     Write Special DEBUG values for IT results
      if(DEBUG) then
         write(iounit,*)
         write(iounit,*) 'ITSET --- IT RESULTS, HOUR :',IHOUR
         IF (SIMPLE) THEN
            write(iounit,*) 'ITFLAG = SIMPLE'
         ELSE IF (INTERM) THEN
            write(iounit,*) 'ITFLAG = INTERM'
         ELSE IF (COMPLX) THEN
            write(iounit,*) 'ITFLAG = COMPLX'
         END IF
         write(iounit,*) 'HECOMP         = ',HECOMP
         write(iounit,*) 'HS, ZS, ZELEV  = ',HS,ZS,ZELEV
      endif

      RETURN
      END

      SUBROUTINE VCALC
C***********************************************************************
C                 VCALC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates concentration or deposition values
C                 for VOLUME sources
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        MODIFIED BY D. Strimaitis, SRC (for Wet & Dry DEPOSITION)
C
C        DATE:    December 15, 1993
C
C        MODIFIED:   To skip receptor if it is more than 80km from source
C                    for TOXICS option.
C                    R. W. Brode, PES - 02/19/99
C
C        MODIFIED:   To allow use with EVENT processing.
C                    R. W. Brode, PES - 12/2/98
C
C        MODIFIED BY D. Strimaitis, SRC (for Dry DEPOSITION)
C        (DATE:    February 15, 1993)
C
C        MODIFIED BY R. Brode, PES, to initialize SBID - 7/15/94
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: 1-hr CONC or DEPOS Values for Each Receptor for
C                 Particular Source
C
C        CALLED FROM:   CALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'VCALC'

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFACT
      CALL EMFACT(QS)

C     Set Deposition Variables for this Source
      IF (LDPART .OR. LWPART .OR. LDGAS) THEN
C        Calculate Deposition Velocities for this Source    ---   CALL VDP
         CALL VDP
      END IF

C     Calculate Scavenging Ratios for this Source           ---   CALL SCAVRAT
      IF (LWPART .OR. LWGAS) CALL SCAVRAT

      IF ((QTK.NE.0.0) .AND. (STABLE .OR. (HS.LE.ZI) .OR.
     &                        DEPOS .OR. WDEP)) THEN
C        Adjust Wind Speed to Release Height                ---   CALL WSADJ
         CALL WSADJ
C        Calculate Effective Radius
         XRAD = 2.15*SYINIT
         IF (LDPART .OR. LWPART) THEN
C           Calculate Min Sigma-z for Settled Plume @ Surface --- CALL SETSZMN
            CALL SETSZMN
         END IF
C        Initialize SBID to 0.0 for call to DEPCOR
         SBID = 0.0

C        Begin Receptor LOOP
         DO IREC = 1, NUMREC
C           Calculate Down and Crosswind Distances          ---   CALL XYDIST
            IF (EVONLY) THEN
               CALL XYDIST(IEVENT)
            ELSE
               CALL XYDIST(IREC)
            END IF
            IF (ABS(Y) .GT. 1.191754*X) THEN
C              Receptor is at least 50 deg. off the plume centerline
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            ELSE IF (DISTR .LT. (XRAD+0.99)) THEN
C              Receptor Too Close to Source for Calculation
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            ELSE IF ((X-XRAD) .LT. 0.0) THEN
C              Receptor Upwind of Downwind Edge
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            ELSE IF (TOXICS .AND. DISTR .GT. 80000.) THEN
C              Receptor is beyond 80km from source.
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            ELSE
C              Determine Effective Plume Height             ---   CALL VHEFF
               CALL VHEFF(ZELEV,HEFLAT,HE)
C              Determine Dispersion Parameters              ---   CALL VDIS
               CALL VDIS(X,SY,SZ,XY,XZ)
               IF (LWGAS) THEN
C                 Initialize wet source depletion factor to unity.
                  WQCORG = 1.
                  IF (WDPLETE) THEN
C                    Determine source depletion factor
C                    from wet removal (GASES)
                     WQCORG=EXP(-GSCVRT*X/US)
                  ENDIF
               ENDIF
C              Calculate Conc. or Depos. for Virtual Point Source
C              Using a Simple Terrain Model                 ---   CALL PSIMPL
               CALL PSIMPL

C              Sum HRVAL to AVEVAL and ANNVAL Arrays        ---   CALL SUMVAL
               IF (EVONLY) THEN
                  CALL EV_SUMVAL
               ELSE
                  CALL SUMVAL
               END IF

            END IF
         END DO
C        End Receptor LOOP
      END IF

      RETURN
      END


      SUBROUTINE ACALC
C***********************************************************************
C                 ACALC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates concentration or deposition values
C                 for AREA sources utilizing an integrated line source.
C
C        PROGRAMMER: Jeff Wang, Roger Brode
C
C        MODIFIED BY D. Strimaitis, SRC (for Wet & Dry DEPOSITION)
C
C        DATE:    November 8, 1993
C
C        MODIFIED:   To modify distance at which model switches to point
C                    source approximation for area sources under TOXICS
C                    option.
C                    R. W. Brode, PES - 02/04/2002
C
C        MODIFIED:   To incorporate optimizations for TOXICS option.
C                    R. W. Brode, PES - 02/19/99
C
C        MODIFIED:   To allow use with EVENT processing and to call
C                    new ARDIST routine instead of XYDIST.
C                    R. W. Brode, PES - 12/2/98
C
C        MODIFIED by YICHENG ZHUANG, SRC to combine version 93188 with
C                 version 93046 - 9/28/93
C
C        MODIFIED:   To incorporate numerical integration algorithm
C                    for AREA source - 7/7/93
C
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION) - 2/15/93
C
C        MODIFIED BY R. Brode, PES, to initialize XZ, XY, and SBID - 7/15/94
C
C*       MODIFIED BY J. Hardikar, PES, to make consistent with the new
C*                   OPENPIT Source Methodology - 7/20/94
C
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Array of 1-hr CONC or DEPOS Values for Each Source/Receptor
C
C        CALLED FROM:   CALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I
      REAL    :: XDEP, WIDTH, LENGTH, XMAXR, QTKSAV, XPOINT
      LOGICAL :: LTERR
      
C     Variable Initializations
      MODNAM = 'ACALC'
C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR = .FALSE.

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFACT
      CALL EMFACT(QS)

C     Set Deposition Variables for this Source
      IF (LDPART .OR. LWPART .OR. LDGAS) THEN
C        Calculate Deposition Velocities for this Source    ---   CALL VDP
         CALL VDP
      END IF

C     Calculate Scavenging Ratios for this Source           ---   CALL SCAVRAT
      IF (LWPART .OR. LWGAS) CALL SCAVRAT

      IF ((QTK.NE.0.0) .AND. (STABLE .OR. (HS.LE.ZI) .OR.
     &                        DEPOS .OR. WDEP)) THEN
C        Adjust Wind Speed to Release Height                ---   CALL WSADJ
         CALL WSADJ
         IF (LDPART .OR. LWPART) THEN
C           Calculate Min Sigma-z for Settled Plume @ Surface --- CALL SETSZMN
            CALL SETSZMN
         END IF
C*       Initialize XY and XZ to 0.0 (XZ is used in
C*       call to DEPCOR from PLUMEF)
         XY = 0.0
         XZ = 0.0
         
C        Initialize SBID to 0.0 (for call to DEPCOR from PLUMEF)
         SBID = 0.0

C        Begin Receptor LOOP
         DO IREC = 1, NUMREC
C           Calculate Down and Crosswind Distances          ---   CALL ARDIST
            IF (EVONLY) THEN
               CALL ARDIST(IEVENT,XDEP,WIDTH,LENGTH,XMAXR)
            ELSE
               CALL ARDIST(IREC,XDEP,WIDTH,LENGTH,XMAXR)
            END IF

C           Check to see if receptor is upwind of area source.
            IF (XMAXR .LT. 1.0) CYCLE

C           Check to see if receptor is more than 80km from source.
            IF (TOXICS .AND. DISTR .GT. 80000.) CYCLE

C           Check to see if receptor is beyond edge of plume laterally.
            IF ( (ABS(Y)-0.5*WIDTH) .GT. 0.) THEN
               CALL ADISY(XMAXR,SY,XY)
               IF ( (ABS(Y)-0.5*WIDTH) .GE. 4.*SY) CYCLE
            END IF

            HE = HS
            HEFLAT = HE
            IF (STABLE .OR. (HEFLAT.LE.ZI) .OR. DEPOS .OR. WDEP) THEN
C              Determine whether area integral or "virtual" point will be used
C              Calculate distance for switch to point source approximation
               XPOINT = 1.5*LENGTH + VP_FACT*WIDTH
               IF (.NOT.TOXICS .OR. SRCTYP(ISRC) .EQ. 'AREAPOLY' .OR.
     &                 (TOXICS .AND. X .LT. XPOINT)) THEN
                  IF (ARDPLETE .AND. LDPART) THEN
C                    Determine deposition correction factors for particles
                     SYINIT = 0.0
                     CALL ADISZ(XDEP,SZ,XZ)
C                    Loop over particle sizes
                     DO I = 1, NPD
C                       Initialize wet & dry source depletion factors, profile
C                       correction factors, and sigma-z settling correction
C                       factors to unity.
                        WQCOR(I)  = 1.0
                        DQCOR(I)  = 1.0
                        PCORZR(I) = 1.0
                        PCORZD(I) = 1.0
                        SZCOR(I)  = 1.0
C                       Determine factors for depletion
C                       from dry removal                    ---   CALL DEPCOR
                        CALL DEPCOR( VDEP(I),VGRAV(I),ZRDEP,ZFLAG,
     &                    XDEP,XZ,HE,ZI,US,XS,YS,XR,YR,
     &                    RURAL,URBAN,KST,SZ,SBID,
     &                    SZMIN(I),ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &                    SRCTYP(ISRC),LTGRID,KURDAT,
     &                    DQCOR(I),PCORZR(I),PCORZD(I),SZCOR(I),TOXICS)
                     END DO
                  ELSE IF (ARDPLETE .AND. LDGAS) THEN
C                    Determine deposition correction factors for particles
                     SYINIT = 0.0
                     CALL ADISZ(XDEP,SZ,XZ)
C                    Initialize source depletion factors to unity.
                     DQCORG  = 1.0
                     PCORZRG = 1.0
                     PCORZDG = 1.0
                     SZCORG  = 1.0
                     WQCORG  = 1.0
C                    Determine factors for depletion - note that
C                    plume ht adjustment for terrain is signalled
C                    by a local logical - LTERR
C                    Simple Terrain Model                   ---   CALL DEPCOR
                     CALL DEPCOR( VDEPG,0.0,ZRDEP,ZFLAG,
     &                 XDEP,XZ,HE,ZI,US,XS,YS,XR,YR,
     &                 RURAL,URBAN,KST,SZ,SBID,
     &                 2.*ZRDEP,ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &                 SRCTYP(ISRC),LTGRID,KURDAT,
     &                 DQCORG,PCORZRG,PCORZDG,SZCORG,TOXICS)
                  END IF
                  DO ITYP = 1, NUMTYP
C                    Calculate Area Source Integral         ---   CALL AREAIN
                     CALL AREAIN
                  END DO
               ELSE
C                 Use point source approximation
C                 Save emissions per unit area and calculate total emissions
                  QTKSAV = QTK
                  IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
                     QTK = QTK * XINIT * YINIT
                  ELSE IF (SRCTYP(ISRC) .EQ. 'AREACIRC') THEN
                     QTK = QTK * PI * RADIUS(ISRC) * RADIUS(ISRC)
                  END IF
                  SYINIT = 0.0
                  CALL VDIS(X,SY,SZ,XY,XZ)
                  CALL PSIMPL
                  QTK = QTKSAV
               END IF

            ELSE
C              Plume Is Above Mixing Height, ZI, and No Wet Deposition
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            END IF

C           Sum HRVAL to AVEVAL and ANNVAL Arrays           ---   CALL SUMVAL
            IF (EVONLY) THEN
               CALL EV_SUMVAL
            ELSE
               CALL SUMVAL
            END IF

         END DO
C        End Receptor LOOP
      END IF

      RETURN
      END


      SUBROUTINE ARDIST(INDX,XDEP,WIDTH,LENGTH,XMAXREC)
C***********************************************************************
C                 ARDIST Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Sets Receptor Variables and Calculates Downwind (X)
C                 and Crosswind (Y) Distances, Crosswind Width (WIDTH),
C                 Distance used for AREADPLT Option (XDEP), Maximum
C                 Downwind Distance by Vertex (XMAXREC), and
C                 Radial Distance from Source to Receptor (DISTR)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Source Location
C                 Arrays of Receptor Locations
C                 SIN and COS of Wind Direction FROM Which Wind
C                 is Blowing, WDSIN and WDCOS
C
C        OUTPUTS: Values of X, Y, and DISTR (m) [in MAIN1]
C                 XDEP (m)
C                 WIDTH (m)
C                 LENGTH (m)
C                 XMAXREC (m)
C
C        CALLED FROM:   ACALC
C                       OCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, INDX
      REAL    :: XSRC, YSRC, XMINREC, XMAXREC, YMINREC, YMAXREC, XDEP,
     &           WIDTH, LENGTH

C     Variable Initializations
      MODNAM = 'ARDIST'

C     Set Receptor Coordinates, Terrain Elevation and Flagpole Heights
      XR = AXR(INDX)
      YR = AYR(INDX)
      ZELEV = AZELEV(INDX)
      ZFLAG = AZFLAG(INDX)

      XMINREC =  9999999.
      XMAXREC = -9999999.
      YMINREC =  9999999.
      YMAXREC = -9999999.

C     Calculate Downwind (X) and Crosswind (Y) Distances for Each Vertex
      DO I = 1, NVERT+1
         XSRC = XVERT(I)
         YSRC = YVERT(I)
         SPA(I,1) = -((XR-XSRC)*WDSIN + (YR-YSRC)*WDCOS)
         SPA(I,2) =   (XR-XSRC)*WDCOS - (YR-YSRC)*WDSIN
         XMINREC = MIN(XMINREC, SPA(I,1))
         XMAXREC = MAX(XMAXREC, SPA(I,1))
         YMINREC = MIN(YMINREC, SPA(I,2))
         YMAXREC = MAX(YMAXREC, SPA(I,2))
      END DO

C     Calculate crosswind width, WIDTH, and alongwind length, LENGTH
      WIDTH  = YMAXREC - YMINREC
      LENGTH = XMAXREC - XMINREC

C     Determine downwind distance to use for AREADPLT option, XDEP
      IF (XMINREC .GE. 0.0) THEN
         XDEP = XMINREC + 0.333333 * LENGTH
      ELSE
         XDEP = 0.333333 * XMAXREC
      END IF

      XDEP = MAX( 1.0, XDEP )

C     Calculate Downwind (X) and Crosswind (Y) Distances from Center of Source
      X = -((XR-XCNTR)*WDSIN + (YR-YCNTR)*WDCOS)
      Y =   (XR-XCNTR)*WDCOS - (YR-YCNTR)*WDSIN

C     Calculate Radial Distance from Center of Source
      DISTR = SQRT(X*X + Y*Y)

      RETURN
      END

      SUBROUTINE OCALC
C***********************************************************************
C                 OCALC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates concentration or deposition values
C                 for OPENPIT sources
C
C        PROGRAMMER: Jayant Hardikar, Roger Brode
C        ADAPTED FROM:  SUBROUTINE ACALC
C
C        DATE:    July 19, 1994
C
C        MODIFIED:   To incorporate optimizations for TOXICS option.
C                    R. W. Brode, PES - 02/19/99
C
C        MODIFIED:   To allow use with EVENT processing.
C                    R. W. Brode, PES - 12/2/98
C
C        MODIFIED:   To skip calculations if QPTOT = 0.0, avoiding
C                    zero divide error in SUB. AMFRAC.
C                    R. W. Brode, PES Inc., - 4/14/95
C
C        INPUTS:  Source Parameters for Specific Source
C                 Arrays of Receptor Locations
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Array of 1-hr CONC or DEPOS Values for Each Source/Receptor
C
C        CALLED FROM:   CALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I, II, ICAT, INOUT
      REAL    :: QPTOT, XVM(5), YVM(5), XDEP, WIDTH, LENGTH, XMAXR,
     &           QTKSAV
      LOGICAL :: LTERR

C     Variable Initializations
      MODNAM = 'OCALC'
C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR = .FALSE.

C     Set the Source Variables for This Source              ---   CALL SETSRC
      CALL SETSRC

C*    Initialize the Total Adjusted Emission Rate from
C*    All Particles
      QPTOT = 0.0          

C*    Loop over Particle Size Categories
      DO ICAT = 1,NPD
C*       Calculate the Escape Fraction for Each Category    ---   CALL ESCAPE      
         CALL ESCAPE(ICAT)

C*       Adjust the Emission Rate for Each Category         ---   CALL ADJEMI
         CALL ADJEMI(ICAT,QPTOT)

C*    End Loop Over Particle Size Categories
      END DO

C*    Skip Calculations if QPTOT = 0.0
      IF (QPTOT .EQ. 0.0)  GO TO 999

C*    Adjust the Mass Fractions for All the Particle 
C*    Size Categories                                       ---   CALL AMFRAC
      CALL AMFRAC(QPTOT)
      
C*    Determine the AlongWind Length of the OPENPIT Source  ---   CALL LWIND
      CALL LWIND

C*    Calculate the Relative Depth of the OPENPIT Source    ---   CALL PDEPTH            
      CALL PDEPTH
      
C*    Calculate the Fractional Size of the 
C*    Effective Pit Area                                    ---   CALL PTFRAC
      CALL PTFRAC


C*    WRITE DEBUG INFORMATION
      IF (DEBUG) THEN
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*) 'DETAILED INFORMATION ON THE OPENPIT SOURCE:'
         WRITE (IOUNIT,*)
         WRITE (IOUNIT,*)   
      ENDIF
     
C*    Determine the Coordinates of the Effective Pit Area
C*    in Wind Direction Coordinate System                   ---   CALL PITEFF
      CALL PITEFF

C*    Calculate the Emission Rate for the Effective
C*    Pit Area                                              ---   CALL PITEMI
      CALL PITEMI(QPTOT)

C*    WRITE DEBUG INFORMATION
      IF (DEBUG) THEN
         WRITE (IOUNIT,*) 'OPENPIT PARTICLE CHARACTERISTICS:'
         WRITE (IOUNIT,*) '---------------------------------'
         WRITE (IOUNIT,*) 
         WRITE (IOUNIT,8000) (EFRAC(II),II = 1, NPD)
8000     FORMAT (1X,'ESCAPE FRACTIONS= ',10(F8.3,2X))
         WRITE (IOUNIT,8200) (QPART(II),II = 1, NPD)
8200     FORMAT (1X,'ADJUSTED EMISSION RATES= ',10(F8.3,2X))
         WRITE (IOUNIT,8400) (PHI(II),II = 1, NPD)
8400     FORMAT (1X,'ADJUSTED MASS FRACTIONS= ',10(F8.3,2X))
         WRITE (IOUNIT,*) 'EMISSION RATE OF EFFECTIVE PIT= ',QEFF
         WRITE (IOUNIT,*) 
      ENDIF                  
      
                  
C     Set Deposition Variables for this Source
      IF (LDPART .OR. LWPART .OR. LDGAS) THEN
C        Calculate Deposition Velocities for this Source    ---   CALL VDP
         CALL VDP
      END IF

C     Calculate Scavenging Ratios for this Source           ---   CALL SCAVRAT
      IF (LWPART .OR. LWGAS) CALL SCAVRAT

C     Apply Variable Emission Rate and Unit Factors         ---   CALL EMFACT
      CALL EMFACT(QEFF)

      IF ((QTK.NE.0.0) .AND. (STABLE .OR. (HS.LE.ZI) .OR.
     &                        DEPOS .OR. WDEP)) THEN
C        Adjust Wind Speed to Release Height                ---   CALL WSADJ
         CALL WSADJ
         IF (LDPART .OR. LWPART) THEN
C           Calculate Min Sigma-z for Settled Plume @ Surface --- CALL SETSZMN
            CALL SETSZMN
         END IF
C        Initialize XY and XZ to 0.0 (XZ is used in call to DEPCOR from PLUMEF)
         XY = 0.0
         XZ = 0.0
C        Initialize SBID to 0.0 (for call to DEPCOR from PLUMEF)
         SBID = 0.0
C        Begin Receptor LOOP
         DO IREC = 1, NUMREC
C           Check for receptor located inside boundary of open pit source
            DO I = 1, NVERT+1
               XVM(I) = AXVERT(I,ISRC)
               YVM(I) = AYVERT(I,ISRC)
            END DO
            XR = AXR(IREC)
            YR = AYR(IREC)
            CALL PNPOLY(XR,YR,XVM,YVM,5,INOUT)
            IF (INOUT .GT. 0) THEN
C              Receptor is within boundary - skip to next receptor
               CYCLE
            END IF

C           Calculate Down and Crosswind Distances          ---   CALL ARDIST
            IF (EVONLY) THEN
               CALL ARDIST(IEVENT,XDEP,WIDTH,LENGTH,XMAXR)
            ELSE
               CALL ARDIST(IREC,XDEP,WIDTH,LENGTH,XMAXR)
            END IF

C           Check it see if receptor is upwind of area source
            IF (XMAXR .LT. 1.0) CYCLE

C           Check to see if receptor is more than 80km from source.
            IF (TOXICS .AND. DISTR .GT. 80000.) CYCLE

            HE = HS
            HEFLAT = HE
            IF (STABLE .OR. (HEFLAT.LE.ZI) .OR. DEPOS .OR. WDEP) THEN
C              Determine whether area integral or "virtual" point will be used
               IF (.NOT.TOXICS .OR.
     &                 (TOXICS .AND. X .LT. VP_FACT*WIDTH)) THEN
                  IF (ARDPLETE .AND. LDPART) THEN
C                    Determine deposition correction factors for particles
                     SYINIT = 0.0
                     CALL ADISZ(XDEP,SZ,XZ)
C                    Loop over particle sizes
                     DO I = 1, NPD
C                       Initialize wet & dry source depletion factors, profile
C                       correction factors, and sigma-z settling correction
C                       factors to unity.
                        WQCOR(I)  = 1.0
                        DQCOR(I)  = 1.0
                        PCORZR(I) = 1.0
                        PCORZD(I) = 1.0
                        SZCOR(I)  = 1.0
C                       Determine factors for depletion
C                       from dry removal                    ---   CALL DEPCOR
                        CALL DEPCOR( VDEP(I),VGRAV(I),ZRDEP,ZFLAG,
     &                    XDEP,XZ,HE,ZI,US,XS,YS,XR,YR,
     &                    RURAL,URBAN,KST,SZ,SBID,
     &                    SZMIN(I),ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &                    SRCTYP(ISRC),LTGRID,KURDAT,
     &                    DQCOR(I),PCORZR(I),PCORZD(I),SZCOR(I),TOXICS)
                     END DO
                  ELSE IF (ARDPLETE .AND. LDGAS) THEN
C                    Determine deposition correction factors for particles
                     SYINIT = 0.0
                     CALL ADISZ(XDEP,SZ,XZ)
C                    Initialize source depletion factors to unity.
                     DQCORG  = 1.0
                     PCORZRG = 1.0
                     PCORZDG = 1.0
                     SZCORG  = 1.0
                     WQCORG  = 1.0
C                    Determine factors for depletion - note that
C                    plume ht adjustment for terrain is signalled
C                    by a local logical - LTERR
C                    Simple Terrain Model                   ---   CALL DEPCOR
                     CALL DEPCOR( VDEPG,0.0,ZRDEP,ZFLAG,
     &                 XDEP,XZ,HE,ZI,US,XS,YS,XR,YR,
     &                 RURAL,URBAN,KST,SZ,SBID,
     &                 2.*ZRDEP,ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &                 SRCTYP(ISRC),LTGRID,KURDAT,
     &                 DQCORG,PCORZRG,PCORZDG,SZCORG,TOXICS)
                  END IF
                  DO ITYP = 1, NUMTYP
C                    Calculate Area Source Integral            ---   CALL AREAIN
                     CALL AREAIN
                  END DO
               ELSE
C                 Use point source approximation
C                 Save emissions per unit area and calculate total emissions
                  QTKSAV = QTK
                  QTK = QTK * XINIT * YINIT
                  SYINIT = 0.0
                  CALL VDIS(X,SY,SZ,XY,XZ)
                  CALL PSIMPL
                  QTK = QTKSAV
               END IF
            ELSE
C              Plume Is Above Mixing Height, ZI, and No Wet Deposition
               DO ITYP = 1, NUMTYP
                  HRVAL(ITYP) = 0.0
                  IF (WETSCIM) HRVALD(ITYP) = 0.0
               END DO
            END IF

C           Sum HRVAL to AVEVAL and ANNVAL Arrays        ---   CALL SUMVAL
            IF (EVONLY) THEN
               CALL EV_SUMVAL
            ELSE
               CALL SUMVAL
            END IF

         END DO
C        End Receptor LOOP
      END IF

 999  RETURN
      END


      SUBROUTINE SETSRC
C***********************************************************************
C                 SETSRC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Sets the Source Parameters for a Particular Source
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        MODIFIED BY: D. Strimaitis, SRC (for Wet & Dry DEPOSITION)
C
C        DATE:  November 8,1993
C
C        MODIFIED by Yicheng Zhuang, SRC to combine version 93188 with
C                 version 93046 - 9/28/93
C
C        MODIFIED:   To incorporate inputs for numerical integration
C                    algorithm for AREA source - 7/7/93
C
C        MODIFIED BY D. Strimaitis, SRC (for DEPOSITION) - 2/15/93
C
C        MODIFIED BY Jayant Hardikar,PES (for handling OPENPIT 
C                    Source - 7/19/94 , also modified AREA Source
C                    for Consistency with OPENPIT Source)
C
C        MODIFIED BY Roger Brode, PES (modified data structure for
C                    AXVERT and AYVERT for consistency with other
C                    2-D source arrays) - 8/15/95
C
C        INPUTS:  Source Parameters Arrays
C                 Source Index
C
C        OUTPUTS: Source Parameters for a Particular Source
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J

C     Variable Initializations
      MODNAM = 'SETSRC'

C     Assign The Values From Array Elements To Variables
      IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         XINIT = AXINIT(ISRC)
         YINIT = AYINIT(ISRC)
         ANGLE = AANGLE(ISRC)
         
         SZINIT = ASZINI(ISRC)         
         NVERT = 4

C        Store Vertices in Temporary Arrays
         DO IVERT = 1, NVERT+1
            XVERT(IVERT) = AXVERT(IVERT,ISRC)
            YVERT(IVERT) = AYVERT(IVERT,ISRC)
         END DO

         XCNTR = AXCNTR(ISRC)
         YCNTR = AYCNTR(ISRC)

      ELSE IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         DS = ADS(ISRC)
         VS = AVS(ISRC)
         TS = ATS(ISRC)

C        Check for Negative Stack Temperature, Used to Indicate Constant TS-TA
         IF (TS .LT. 0.0) THEN
            TS = TA + ABS(TS)
         END IF

         IF (IFVSEC .LE. NSEC) THEN
            DSBH = ADSBH(IFVSEC,ISRC)
            DSBW = ADSBW(IFVSEC,ISRC)
            IF (IDSWAK(IFVSEC,ISRC) .EQ. 0) THEN
               WAKLOW = .FALSE.
            ELSE IF (IDSWAK(IFVSEC,ISRC) .EQ. 1) THEN
               WAKLOW = .TRUE.
            END IF
         END IF

      ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         SYINIT = ASYINI(ISRC)
         SZINIT = ASZINI(ISRC)

      ELSE IF (SRCTYP(ISRC) .EQ. 'AREA') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         XINIT = AXINIT(ISRC)
         YINIT = AYINIT(ISRC)
         ANGLE = AANGLE(ISRC)
         
         SZINIT = ASZINI(ISRC)         
         NVERT = 4

C        Store Vertices in Temporary Arrays
         DO IVERT = 1, NVERT+1
            XVERT(IVERT) = AXVERT(IVERT,ISRC)
            YVERT(IVERT) = AYVERT(IVERT,ISRC)
         END DO

         XCNTR = AXCNTR(ISRC)
         YCNTR = AYCNTR(ISRC)

      ELSE IF (SRCTYP(ISRC) .EQ. 'AREAPOLY') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         SZINIT = ASZINI(ISRC)
         NVERT  = NVERTS(ISRC)

C        Store Vertices in Temporary Arrays
         DO IVERT = 1, NVERT+1
            XVERT(IVERT) = AXVERT(IVERT,ISRC)
            YVERT(IVERT) = AYVERT(IVERT,ISRC)
         END DO
            
         XCNTR = AXCNTR(ISRC)
         YCNTR = AYCNTR(ISRC)

      ELSE IF (SRCTYP(ISRC) .EQ. 'AREACIRC') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
         HS = AHS(ISRC)

         SZINIT = ASZINI(ISRC)
         NVERT  = NVERTS(ISRC)

C        Store Vertices in Temporary Arrays
         DO IVERT = 1, NVERT+1
            XVERT(IVERT) = AXVERT(IVERT,ISRC)
            YVERT(IVERT) = AYVERT(IVERT,ISRC)
         END DO
            
         XCNTR = AXCNTR(ISRC)
         YCNTR = AYCNTR(ISRC)

      ELSE IF (SRCTYP(ISRC) .EQ. 'OPENPIT') THEN
         XS = AXS(ISRC)
         YS = AYS(ISRC)
         ZS = AZS(ISRC)
         QS = AQS(ISRC)
C        Set Emission Height of Effective Area, HS = 0.0
         HS = 0.0
C        Set Height of Emissions Above Base of Pit, EMIHGT
         EMIHGT = AHS(ISRC)
         NVERT = 4

         XINIT = AXINIT(ISRC)
         YINIT = AYINIT(ISRC)
         ANGLE = AANGLE(ISRC)
         PALPHA = AALPHA(ISRC)
         PDEFF  = APDEFF(ISRC)
         SZINIT = ASZINI(ISRC)
         PITLEN = MAX(XINIT,YINIT)
         PITWID = MIN(XINIT,YINIT)

C        Store Vertices in Temporary Arrays
         DO IVERT = 1, NVERT+1
            XVERT(IVERT) = AXVERT(IVERT,ISRC)
            YVERT(IVERT) = AYVERT(IVERT,ISRC)
         END DO
              
         XCNTR = AXCNTR(ISRC)
         YCNTR = AYCNTR(ISRC)

      END IF

      NPD = INPD(ISRC)
      IF (NPD .GT. 0) THEN
         DO J = 1, NPD
            PDIAM(J) = APDIAM(J,ISRC)
            PHI(J) = APHI(J,ISRC)
            PDENS(J) = APDENS(J,ISRC)
            VGRAV(J) = AVGRAV(J,ISRC)
            TSTOP(J) = ATSTOP(J,ISRC)
            SC(J) = ASC(J,ISRC)
            PSCAV(J,1) = APSLIQ(J,ISRC)
            PSCAV(J,2) = APSICE(J,ISRC)
         END DO
      END IF

C     Transfer Gas Wet Scavenging Coeff. (1:liquid, 2:frozen)
      GSCAV(1) = AGSCAV(1,ISRC)
      GSCAV(2) = AGSCAV(2,ISRC)

      RETURN
      END

      SUBROUTINE PHEFF(XARG,DHPOUT,HEOUT)
C***********************************************************************
C                 PHEFF Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Effective Plume Height for POINT Sources (m)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To remove terrain adjustment to separate subroutine,
C                    and to use calling arguments
C                    R.W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C                 Terrain Elevation of Receptor
C
C        OUTPUTS: Plume Height (HEOUT) without Terrain Adjustment
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: HEOUT, DHPOUT, XARG, HSPRIM

C     Variable Initializations
      MODNAM = 'PHEFF'

C     Calculate Plume Height Without Terrain Adjustment
      IF ((.NOT. WAKE) .AND. (.NOT. GRDRIS)) THEN
C        Calculate Final Rise for First Receptor Only
         IF (FSTREC) THEN
            FSTREC = .FALSE.
            HSP = HSPRIM(US,VS,HS,DS)
C           Calculate Final Rise, DHF                       ---   CALL DELH
            CALL DELH(DHF)
         END IF
         IF (NOSTD) THEN
            HEOUT = HS + DHF
         ELSE
            HEOUT = HSP + DHF
         END IF
         IF (.NOT. NOBID) THEN
C           Calculate Gradual Plume Rise for Use in BID Calculation
            IF (XARG .LT. XF) THEN
C              Calculate Gradual Rise, DHPOUT               ---   CALL DHPHS
               CALL DHPHS(XARG,DHF,DHPOUT)
            ELSE
               DHPOUT = DHF
            END IF
         ELSE
            DHPOUT = DHF
         END IF
      ELSE IF (WAKE .AND. WAKESS) THEN
C        Calculate Final Rise for First Receptor Only
         IF (FSTREC) THEN
            FSTREC = .FALSE.
C           Calculate Final Rise (at X=XF), DHF             ---   CALL DHPSS
            CALL DHPSS(XF,DHPOUT)
            DHF    = DHPOUT
         END IF
         IF (XARG .LT. XF) THEN
C           Calculate Gradual Rise, DHP                     ---   CALL DHPSS
            CALL DHPSS(XARG,DHPOUT)
         ELSE
            DHPOUT = DHF
         END IF
         HEOUT = HS + DHPOUT
      ELSE
CRWB       if ((WAKE .AND. (.NOT. WAKESS)) .OR.
CRWB          ((.NOT. WAKE) .AND. GRDRIS)) then
C        Calculate Final Rise for First Receptor Only
         IF (FSTREC) THEN
            FSTREC = .FALSE.
            HSP = HSPRIM(US,VS,HS,DS)
C           Calculate Final Rise, DHF                       ---   CALL DELH
            CALL DELH(DHF)
         END IF
         IF (XARG .LT. XF) THEN
C           Calculate Gradual Rise, DHP                     ---   CALL DHPHS
            CALL DHPHS(XARG,DHF,DHPOUT)
         ELSE
            DHPOUT = DHF
         END IF
         IF (NOSTD) THEN
            HEOUT = HS + DHPOUT
         ELSE
            HEOUT = HSP + DHPOUT
         END IF
      END IF

      RETURN
      END

      SUBROUTINE PHEFFC(XARG,DHPOUT,HEOUT)
C***********************************************************************
C                 PHEFFC Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Effective Plume Height for POINT Sources (m)
C                 in Complex Terrain
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    September 30, 1994
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C                 Terrain Elevation of Receptor
C
C        OUTPUTS: Plume Height (HEOUT) without Terrain Adjustment
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: HEOUT, DHPOUT, XARG, HSPRIM

C     Variable Initializations
      MODNAM = 'PHEFFC'

      IF (FSTCMP) THEN
         FSTCMP = .FALSE.
C        This is the First Call for PHEFFC - Calculate HSP and DHFCMP
         HSP = HSPRIM(US,VS,HS,DS)
         CALL DELH(DHFCMP)
      END IF

      IF (XARG .LT. XF) THEN
C        Distance is less than distance to final rise - Calculate gradual rise
         CALL DHPHS(XARG,DHFCMP,DHPOUT)
      ELSE
C        Set gradual rise = final rise for XARG > XF
         DHPOUT = DHFCMP
      END IF

C     Check for stack-tip downwash option
      IF (NOSTD) THEN
         HEOUT = HS + DHPOUT
      ELSE
         HEOUT = HSP + DHPOUT
      END IF

      RETURN
      END

      SUBROUTINE STERAD(HEARG,ZARG,HEOUT)
C***********************************************************************
C                 STERAD Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Adjusts Effective Plume Height for Simple Terrain Effects
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    September 30, 1994
C
C        INPUTS:  HEARG = Flat terrain plume height
C                 ZARG  = Elevation of terrain
C
C        OUTPUTS: HEOUT = Effective plume height with terrain adjustment
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: HEOUT, HEARG, ZARG, HTERCHOP

C     Variable Initializations
      MODNAM = 'STERAD'

C     Adjust Plume Height for Elevated Terrain, Save Flat Terrain Value (HEFLAT)
C     For Later Comparison With Mixing Height
      IF (FLAT) THEN
         HEOUT  = HEARG
      ELSE IF (ELEV) THEN
C        Calculate Terrain Hgt Above Plant Grade (Chopped-off at Release Height)
         HTERCHOP = MIN( HS, (ZARG - ZS))
         HEOUT = HEARG - HTERCHOP
      END IF

C     Don't Allow Effective Plume Height to be < 0.0
      HEOUT = MAX( 0.0, HEOUT)

      RETURN
      END

      SUBROUTINE CTERAD(HEARG,ZARG,HEOUT,COUT)
C***********************************************************************
C                 CTERAD Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Adjusts Effective Plume Height for Complex Terrain Effects
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    September 30, 1994
C
C        INPUTS:  HEARG = Flat terrain plume height
C                 ZARG  = Elevation of terrain
C
C        OUTPUTS: HEOUT = Effective plume height with terrain adjustment
C                 COUT  = Attenuation correction factor
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: COUT, HEOUT, ZARG, HEARG

C     Variable Initializations
      MODNAM = 'CTERAD'

C     Calculate Terrain Hgt Above Plant Grade
      HTER = ZARG - ZS

C     Calculate COMPLEX1 Plume Height
      HEOUT = MAX( (HEARG*TCF(KST)),
     &               (HEARG-(1.0-TCF(KST))*HTER) )
      HEOUT = MAX( HEOUT, ZMIN )

C     Calculate the Attentuation Correction Factor, COUT
      IF ( (UNSTAB.OR.NEUTRL) .OR. (HEARG.GE.(HTER+ZFLAG)) ) THEN
         COUT = 1.0
      ELSE IF ((HTER+ZFLAG-HEARG) .GE. 400.) THEN
         COUT = 0.0
      ELSE
         COUT = (400. - (HTER+ZFLAG-HEARG))/400.
      END IF

      RETURN
      END

      SUBROUTINE VHEFF(ZARG,HEFOUT,HEOUT)
C***********************************************************************
C                 VHEFF Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Effective Plume Height for VOLUME Sources (m)
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Meteorological Variables for One Hour
C                 Wind Speed Adjusted to Stack Height
C                 Downwind Distance
C                 Terrain Elevation of Receptor
C
C        OUTPUTS: Effective Plume Height (HE)
C
C        CALLED FROM:   VCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: HEOUT, HEFOUT, ZARG, HTERCHOP

C     Variable Initializations
      MODNAM = 'VHEFF'

C     Calculate Terrain Height Above Plant Grade (Chopped-off at Release Height)
      IF (FLAT) THEN
         HTERCHOP = 0.0
      ELSE IF (ELEV) THEN
         HTERCHOP = MIN( HS, (ZARG - ZS))
      END IF

C     Calculate Effective Plume Height (No Rise) Adjusted for Terrain Height
      HEOUT = HS - HTERCHOP

C     Save Plume Height for Flat Terrain for Later Comparison to Mixing Height
      HEFOUT = HS

      RETURN
      END

      SUBROUTINE PDIS(XARG,SYOUT,SZOUT,XYOUT,XZOUT,SBOUT)
C***********************************************************************
C                 PDIS Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Dispersion Parameters for POINT Sources
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C        MODIFIED BY D. Strimaitis, SRC (initialize SBID to 0.0)
C
C        DATE:    February 15, 1993
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Wake Plume Height, HEMWAK
C                 Meteorological Variables for One Hour
C                 Downwind Distance
C
C        OUTPUTS: Lateral and Vertical Dispersion Coefficients, SY and SZ
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: SBOUT, XYOUT, XZOUT, SYOUT, SZOUT, XARG, SYARG, SZARG

C     Variable Initializations
      MODNAM = 'PDIS'

      IF (.NOT. WAKE) THEN
C        Calculate Sigma-y from Curves                   ---   CALL SIGY
         CALL SIGY(XARG,SYARG)
C        Calculate Sigma-z from Curves                   ---   CALL SIGZ
         CALL SIGZ(XARG,SZARG)
         IF (.NOT. NOBID) THEN
C           Apply BID                                    ---   CALL BID
            CALL BID(DHP,SYARG,SZARG,SYOUT,SZOUT,SBOUT)
         ELSE
            SBOUT = 0.0
            SYOUT = SYARG
            SZOUT = SZARG
         END IF
         XYOUT = 0.0
         XZOUT = 0.0
      ELSE IF (WAKE) THEN
         IF (HEMWAK .GT. 1.2*DSBH) THEN
C           Calculate Sigma-y from Curves                ---   CALL SIGY
            CALL SIGY(XARG,SYARG)
            XYOUT = 0.0
         ELSE
C           Calculate Building Enhanced Sigma-y          ---   CALL SYENH
            CALL SYENH(XARG,SYARG,XYOUT)
         END IF
C        Calculate Building Enhanced Sigma-z             ---   CALL SZENH
         CALL SZENH(XARG,SZARG,XZOUT)
         IF ((.NOT. NOBID) .AND. (.NOT. WAKESS)) THEN
C           Apply BID                                    ---   CALL BID
            CALL BID(DHP,SYARG,SZARG,SYOUT,SZOUT,SBOUT)
         ELSE
            SBOUT = 0.0
            SYOUT = SYARG
            SZOUT = SZARG
         END IF
      END IF

      IF (SZOUT .GT. 5000. .AND. NPD .EQ. 0)  SZOUT = 5000.

      RETURN
      END

      SUBROUTINE PDISC(XARG,SZOUT,XZOUT,SBOUT)
C***********************************************************************
C                 PDISC Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Dispersion Parameters for POINT Sources
C
C        PROGRAMMER: Roger Brode
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Arrays of Source Parameters
C                 Logical Wake Flags
C                 Wake Plume Height, HEMWAK
C                 Meteorological Variables for One Hour
C                 Downwind Distance
C
C        OUTPUTS: Lateral and Vertical Dispersion Coefficients, SY and SZ
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: SBOUT, XZOUT, SYOUT, SZOUT, XARG, SYARG, SZARG

C     Variable Initializations
      MODNAM = 'PDISC'

C     Calculate Sigma-z from Curves Using Radial Distance   ---   CALL SIGZ
      CALL SIGZ(XARG,SZARG)
      SYARG = 0.0

      IF (.NOT. NOBID) THEN
C        Apply BID                                          ---   CALL BID
         CALL BID(DHPCMP,SYARG,SZARG,SYOUT,SZOUT,SBOUT)
      ELSE
         SBOUT = 0.0
         SZOUT = SZARG
         SYOUT = 0.0
      END IF
      XZOUT = 0.0

      IF (SZOUT .GT. 5000. .AND. NPD .EQ. 0)  SZOUT = 5000.

      RETURN
      END

      SUBROUTINE VDIS(XARG,SYOUT,SZOUT,XYOUT,XZOUT)
C***********************************************************************
C                 VDIS Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Dispersion Parameters for VOLUME Sources
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        MODIFIED:   To remove input distance calling argument for XVZ
C                    R. W. Brode, PES, Inc. - 12/29/97
C
C        INPUTS:  Arrays of Source Parameters
C                 Meteorological Variables for One Hour
C                 Downwind Distance
C
C        OUTPUTS: Lateral and Vertical Dispersion Coefficients
C
C        CALLED FROM:   VCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: XYOUT, XZOUT, SYOUT, SZOUT, XARG

C     Variable Initializations
      MODNAM = 'VDIS'

C     Calculate Lateral Virtual Distance                 ---   CALL XVY
      CALL XVY(XYOUT)
C     Calculate Sigma-y from Curves for X+XY             ---   CALL SIGY
      CALL SIGY(XARG+XYOUT,SYOUT)
C     Calculate Vertical Virtual Distance                ---   CALL XVZ
      CALL XVZ(XZOUT)
C     Calculate Sigma-z from Curves for X+XZ             ---   CALL SIGZ
      CALL SIGZ(XARG+XZOUT,SZOUT)

      IF (SZOUT .GT. 5000. .AND. NPD .EQ. 0)  SZOUT = 5000.

      RETURN
      END

      SUBROUTINE ADISY(XARG,SYOUT,XYOUT)
C***********************************************************************
C                 ADISY Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Lateral Dispersion Parameters for AREA Sources
C
C        PROGRAMMER: Roger Brode, PES, Inc.
C
C        DATE:    December 14, 1998
C
C        INPUTS:  Downwind Distance (m), XARG
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Lateral Dispersion Coefficient (m), SYOUT
C                 Lateral Virtual Distance (m), XYOUT
C
C        CALLED FROM:   PLUMEF, PWIDTH
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: XARG, SYOUT, XYOUT

C     Variable Initializations
      MODNAM = 'ADISY'

C     Calculate Sigma-y from Curves for XARG                ---   CALL SIGY
      CALL SIGY(XARG,SYOUT)
      SYOUT = MAX(SYOUT,0.0001)
      XYOUT = 0.0

      RETURN
      END

      SUBROUTINE ADISZ(XARG,SZOUT,XZOUT)
C***********************************************************************
C                 ADISZ Module of the ISC Short Term Model - Version 2
C
C        PURPOSE: Calculates Vertical Dispersion Parameters for AREA Sources
C
C        PROGRAMMER: Roger Brode, PES, Inc.
C
C        DATE:    December 14, 1998
C
C        INPUTS:  Downwind Distance (m), XARG
C                 Meteorological Variables for One Hour
C
C        OUTPUTS: Vertical Dispersion Coefficient (m), SZOUT
C                 Vertical Virtual Distance (m), XZOUT
C
C        CALLED FROM:   PLUMEF
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: XARG, SZOUT, XZOUT

C     Variable Initializations
      MODNAM = 'ADISZ'

C     Calculate Sigma-z from Curves for XARG                ---   CALL SIGZ
      CALL SIGZ(XARG,SZOUT)
      SZOUT = MAX(SZOUT,0.0001)
      XZOUT = 0.0

C     Add Initial Dispersion for OPENPIT Sources
      IF (SZINIT .GT. 0.0) THEN
         SZOUT = SQRT (SZOUT*SZOUT + SZINIT*SZINIT)
      END IF

      IF (SZOUT .GT. 5000. .AND. NPD .EQ. 0)  SZOUT = 5000.

      RETURN
      END

      SUBROUTINE PSIMPL
C***********************************************************************
C               PSIMPL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Hourly Concentration or Deposition
C                 value for POINT Sources
C                 Using Gaussian Plume Equation for Simple Terrain
C
C                 (Replaces PCHI and PDEP)
C
C           NOTE: Particle settling is treated as a "tilted plume"
C                 until the centerline reaches the surface.  Thereafter
C                 the centroid height of the plume continues to be
C                 modified by gravity.  This process is simulated by
C                 altering the sigma-z for each particle-size.  Hence,
C                 sigma-z is now a function of particle-size.
C
C        PROGRAMMER: D. Strimaitis, SRC
C
C        DATE:    December 15, 1993
C
C        MODIFIED:   To compare YTERM to -18.0 rather than EXPLIM,
C                    equivalent to 6*SY.  R.W. Brode, PES, Inc. - 02/19/99
C
C        MODIFIED:   To call PDEP for call to SUB. DEPCOR; to use
C                    modified SUB. VERT.  R.W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:
C
C        OUTPUTS: HRVAL, Concentration or Deposition for Particular
C                 Source/Receptor Combination
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J
      REAL :: YTERM, VSIMP, A0, SZADJ, HV, ADJ, VJ, DRYFLUX, WETFLUX,
     &        VTERM, VSIMPD, VTMP
      LOGICAL WDONLY

C     Variable Initializations
      MODNAM = 'PSIMPL'
      WDONLY = .FALSE.

      IF ((UNSTAB .OR. NEUTRL) .AND. HEFLAT.GT.ZI) THEN
C        Plume Is Above Mixing Height, ZI
         IF (DEPOS .OR. WDEP) THEN
C           Set WDONLY flag for Wet Deposition Only
            WDONLY = .TRUE.
         ELSE
            DO ITYP = 1, NUMTYP
               HRVAL(ITYP) = 0.0
               IF (WETSCIM) HRVALD(ITYP) = 0.0
            END DO
            RETURN
         END IF
      END IF

      YTERM = -0.5*(Y*Y)/(SY*SY)

C     If receptor is more than 6.*SY off the centerline (YTERM < -18.),
C     then skip calculation, otherwise continue.
      IF (YTERM .GT. -18.0) THEN

         IF (NPD .EQ. 0) THEN
C           Determine Deposition Correction Factors for Gases
            IF (LDGAS .OR. LWGAS) THEN
               CALL PDEPG (X, WDONLY)
            ENDIF
            DO ITYP = 1, NUMTYP
               V(ITYP) = 0.
               IF (WETSCIM) VDRY(ITYP) = 0.
            END DO
            VSIMP = 0.0
            IF (WETSCIM) VSIMPD = 0.0
            ITYP = 0
            A0  = -0.5/(SZ*SZ)
            ADJ = DQCORG * WQCORG
            IF (CONC) THEN
C              Concentration
               ITYP = ITYP + 1
               IF (WDONLY) THEN
C                 Plume is above mixing height so set CONC = 0.0
                  V(ITYP) = 0.0
               ELSE
C                 Calculate Concentration Form of V         ---   CALL VERT
                  CALL VERT(HE,SZ,A0,ZFLAG,V(ITYP))
                  V(ITYP) = ADJ*PCORZRG*V(ITYP)/SZ
                  IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
               END IF
            END IF
            IF (DEPOS .OR. DDEP) THEN
               IF (WDONLY) THEN
C                 Plume is above mixing height so set DDEP = 0.0
                  DRYFLUX = 0.0
               ELSE
C                 For Dry Deposition Complete Vertical Term is Needed
C                 Calculated at ZRDEP                       ---   CALL VERT
                  CALL VERT(HE,SZ,A0,ZRDEP,VD)
                  DRYFLUX = ADJ*PCORZDG*VDEPG*VD/SZ
               END IF
            END IF
            IF (DEPOS .OR. WDEP) THEN
C              Calculate Wet Flux
C              For Wet Flux, Vertical Term is Integral of EXP terms
C              Over All z, so VJ/SZ=SQRT(2PI)
               WETFLUX = ADJ*GSCVRT*SRT2PI
            ENDIF
            IF (DEPOS) THEN
C              Wet & Dry fluxes of particles are summed
               ITYP = ITYP + 1
               V(ITYP) = DRYFLUX + WETFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
            END IF
            IF (DDEP) THEN
C              Dry flux of particles
               ITYP = ITYP + 1
               V(ITYP) = DRYFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
            END IF
            IF (WDEP) THEN
C              Wet flux of particles
               ITYP = ITYP + 1
               V(ITYP) = WETFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
            ENDIF
            IF (.NOT.CONC .AND. INTERM) THEN
               IF (WDONLY) THEN
C                 Plume is above mixing height so set CONC = 0.0
                  VSIMP = 0.0
               ELSE
C                 For Concentration Complete Vertical Term is Needed for
C                 Each Particle Size Calculated at ZFLAG ---   CALL VERT
                  CALL VERT(HE,SZ,A0,ZFLAG,VSIMP)
C                 Calculate Concentration for Intermediate Terrain Check
                  VSIMP = ADJ*PCORZRG*VSIMP/SZ
                  IF (WETSCIM) VSIMPD = VSIMP / WQCORG
               END IF
            END IF

         ELSE
C           Determine Deposition Correction Factors for Particles
            IF (LDPART .OR. LWPART) THEN
               CALL PDEP (X, WDONLY)
            ENDIF

C           Calculate the Vertical Term, V for particles
            DO ITYP = 1, NUMTYP
               V(ITYP) = 0.
               IF (WETSCIM) VDRY(ITYP) = 0.
            END DO
            VSIMP = 0.
            IF (WETSCIM) VSIMPD = 0.
            DO J = 1, NPD
               ITYP = 0
C              Settling may alter SZ for the Jth particle plume
               SZADJ = SZ*SZCOR(J)
               A0 = -0.5/(SZADJ*SZADJ)
C              Calculate Plume Tilt Due to Settling, HV
               HV = (X/US) * VGRAV(J)
C              Calculate Settled Plume Height, HESETL
               HESETL = HE - HV
C              Restrict settled height to be positive, so that the plume
C              does not settle below the surface -- this is the limit of
C              the tilted plume technique.
               HESETL = MAX(0.0,HESETL)
C              Adjust Jth contribution by mass fraction and source
C              depletion
               ADJ = PHI(J) * DQCOR(J) * WQCOR(J)
               IF (CONC) THEN
C                 Concentration
                  ITYP = ITYP + 1
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set CONC = 0.0
                     V(ITYP) = 0.0
                  ELSE
C                    For Concentration Complete Vertical Term is Needed for
C                    Each Particle Size Calculated at ZFLAG ---   CALL VERT
                     CALL VERT(HESETL,SZADJ,A0,ZFLAG,VJ)
                     VTMP = ADJ*PCORZR(J)*VJ/SZADJ
                     V(ITYP) = V(ITYP) + VTMP
                     IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP) +VTMP/WQCOR(J)
                  END IF
               END IF
               IF (DEPOS .OR. DDEP) THEN
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set DDEP = 0.0
                     DRYFLUX = 0.0
                  ELSE
C                    For Dry Deposition Complete Vertical Term is Needed for
C                    Each Particle Size Calculated at ZRDEP ---   CALL VERT
                     CALL VERT(HESETL,SZADJ,A0,ZRDEP,VJ)
C                    Calculate Dry Flux VJ/SZ
                     DRYFLUX = ADJ*PCORZD(J)*VDEP(J)*VJ/SZADJ
                  END IF
               END IF
               IF (DEPOS .OR. WDEP) THEN
C                 Calculate Wet Flux VJ/SZ --
C                 For Wet Flux, Vertical Term is Integral of EXP terms
C                 Over All z, so VJ/SZ=SQRT(2PI)
                  WETFLUX = ADJ*PSCVRT(J)*SRT2PI
               ENDIF
               IF (DEPOS) THEN
C                 Wet & Dry fluxes of particles are summed
                  ITYP = ITYP + 1
                  VTMP = DRYFLUX + WETFLUX
                  V(ITYP) = V(ITYP) + VTMP
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP) + VTMP/WQCOR(J)
               END IF
               IF (DDEP) THEN
C                 Dry flux of particles
                  ITYP = ITYP + 1
                  V(ITYP) = V(ITYP) + DRYFLUX
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP) +DRYFLUX/WQCOR(J)
               END IF
               IF (WDEP) THEN
C                 Wet flux of particles
                  ITYP = ITYP + 1
                  V(ITYP) = V(ITYP) + WETFLUX
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP) +WETFLUX/WQCOR(J)
               ENDIF
               IF (.NOT.CONC .AND. INTERM) THEN
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set CONC = 0.0
                     VSIMP = 0.0
                  ELSE
C                    For Concentration Complete Vertical Term is Needed for
C                    Each Particle Size Calculated at ZFLAG ---   CALL VERT
                     CALL VERT(HESETL,SZADJ,A0,ZFLAG,VJ)
C                    Calculate Concentration for Intermediate Terrain Check
                     VTMP  = ADJ*PCORZR(J)*VJ/SZADJ
                     VSIMP = VSIMP + VTMP
                     IF (WETSCIM) VSIMPD = VSIMPD + VTMP / WQCOR(J)
                  END IF
               END IF
            ENDDO
         END IF

C        Calculate the Decay Term, D                     ---   CALL DECAY
         CALL DECAY (X)

         DO ITYP = 1, NUMTYP
C           Complete VTERM (SZ already in denomenator of V)
            VTERM = (D*V(ITYP))/(TWOPI*US*SY)

C           Check for Possible Underflow Condition
            IF (VTERM.GT.0.0 .AND. (LOG(VTERM)+YTERM).GT.EXPLIM) THEN
               HRVAL(ITYP) = QTK * EMIFAC(ITYP) * VTERM * EXP(YTERM)
            ELSE
               HRVAL(ITYP) = 0.0
            END IF
            IF (WETSCIM) THEN
C              Repeat the above calculations for HRVALD array

               VTERM = (D*VDRY(ITYP))/(TWOPI*US*SY)
               IF (VTERM.GT.0.0 .AND. (LOG(VTERM)+YTERM).GT.EXPLIM) THEN
                  HRVALD(ITYP) = QTK * EMIFAC(ITYP) * VTERM * EXP(YTERM)
               ELSE
                  HRVALD(ITYP) = 0.0
               END IF
            ENDIF
         END DO

         IF (.NOT.CONC .AND. INTERM) THEN
C           Calculate Concentration for Simple Terrain
C           Complete VTERM (SZ already in denomenator of V)
            VTERM = (D*VSIMP)/(TWOPI*US*SY)

C           Check for Possible Underflow Condition
            IF (VTERM.GT.0.0 .AND. (LOG(VTERM)+YTERM).GT.EXPLIM) THEN
               SIMCON = QTK * EMICON * VTERM * EXP(YTERM)
            ELSE
               SIMCON = 0.0
            END IF
         ELSE IF (CONC .AND. INTERM) THEN
            SIMCON = HRVAL(1)
         END IF

      ELSE
C        Lateral Term is 0.0; Set HRVAL's to 0.0
         DO ITYP = 1, NUMTYP
            HRVAL(ITYP) = 0.0
            IF (WETSCIM) HRVALD(ITYP) = 0.0
         END DO
         SIMCON = 0.0
      END IF

      IF (DEBUG) THEN
C        Print Out Debugging Information                    ---   CALL DEBOUT
         CALL DEBOUT
      END IF

 999  RETURN
      END

      SUBROUTINE PCOMPL
C***********************************************************************
C               PCOMPL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Hourly Concentration or Deposition
C                 value for POINT Sources
C                 Using Gaussian Plume Equation for Complex Terrain
C
C                 (Replaces PCHI and PDEP)
C
C           NOTE: Particle settling is treated as a "tilted plume"
C                 until the centerline reaches the surface.  Thereafter
C                 the centroid height of the plume continues to be
C                 modified by gravity.  This process is simulated by
C                 altering the sigma-z for each particle-size.  Hence,
C                 sigma-z is now a function of particle-size.
C
C        PROGRAMMER: D. Strimaitis, SRC
C
C        DATE:    December 15, 1993
C
C        MODIFIED:   To use fully integrated COMPLEX1 algorithms rather
C                    than calls to CMP1.  R.W. Brode, PES, Inc. - 9/30/94
C
C        INPUTS:
C
C        OUTPUTS: HRVAL, Concentration or Deposition for Particular
C                 Source/Receptor Combination
C
C        CALLED FROM:   PCALC
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J
      REAL :: VCOMP, A0, SZADJ, HV, CORRJ, ADJ, VJ, DRYFLUX, WETFLUX,
     &        WCMP1, VCOMPD, VTMP
      LOGICAL WDONLY

C     Variable Initializations
      MODNAM = 'PCOMPL'
      WDONLY = .FALSE.

      IF ((UNSTAB .OR. NEUTRL) .AND. HECOMP.GT.ZI) THEN
C        Plume Is Above Mixing Height, ZI
         IF (DEPOS .OR. WDEP) THEN
C           Set WDONLY flag for Wet Deposition Only
            WDONLY = .TRUE.
         ELSE
            DO ITYP = 1, NUMTYP
               HRVAL(ITYP) = 0.0
               IF (WETSCIM) HRVALD(ITYP) = 0.0
            END DO
            RETURN
         END IF
      END IF

      IF (ABS(Y).LE.X*0.19891 .AND. CORR.GT.0.0) THEN
C        Receptor is inside of sector and Plume is < 400m Below Receptor

         IF (NPD .EQ. 0) THEN
C           Determine Deposition Correction Factors for Gases
            IF (LDGAS .OR. LWGAS) THEN
               CALL PDEPGC (WDONLY)
            ENDIF
            DO ITYP = 1, NUMTYP
               V(ITYP) = 0.
               IF (WETSCIM) VDRY(ITYP) = 0.
            END DO
            VCOMP = 0.0
            IF (WETSCIM) VCOMPD = 0.0
            ITYP = 0
            A0 = -0.5/(SZCMP1*SZCMP1)
            ADJ = DQCORGC * WQCORGC
            IF (CONC) THEN
C              Concentration
               ITYP = ITYP + 1
               IF (WDONLY) THEN
C                 Plume is above mixing height so set CONC = 0.0
                  V(ITYP) = 0.0
                  IF (WETSCIM) VDRY(ITYP) = 0.0
               ELSE
C                 For Concentration, Complete Vertical Term is Needed for
C                 Each Particle Size Calculated at ZFLAG ---   CALL VERT
                  CALL VERT(HECMP1,SZCMP1,A0,ZFLAG,V(ITYP))
                  V(ITYP) = CORR*ADJ*PCORZRGC*V(ITYP)/SZCMP1
                  IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORGC
               END IF
            END IF
            IF (DEPOS .OR. DDEP) THEN
               IF (WDONLY) THEN
C                 Plume is above mixing height so set DRYFLUX = 0.0
                  DRYFLUX = 0.0
               ELSE
C                 For Dry Deposition, Complete Vertical Term is Needed for
C                 Each Particle Size Calculated at ZRDEP ---   CALL VERT
                  CALL VERT(HECMP1,SZCMP1,A0,ZRDEP,VD)
C                 Calculate Dry Flux VJ/SZ
                  DRYFLUX = CORR*ADJ*PCORZDGC*VDEPG*VD/SZCMP1
               END IF
            END IF
            IF (DEPOS .OR. WDEP) THEN
C              Calculate Wet Flux VJ/SZ --
C              For Wet Flux, Vertical Term is Integral of EXP terms
C              Over All z, so VJ/SZ=SQRT(2PI)
               WETFLUX = CORR*ADJ*GSCVRT*SRT2PI
            ENDIF
            IF (DEPOS) THEN
C              Wet & Dry fluxes of particles are summed
               ITYP = ITYP + 1
               V(ITYP) = DRYFLUX + WETFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORGC
            END IF
            IF (DDEP) THEN
C              Dry flux of particles
               ITYP = ITYP + 1
               V(ITYP) = DRYFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORGC
            END IF
            IF (WDEP) THEN
C              Wet flux of particles
               ITYP = ITYP + 1
               V(ITYP) = WETFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORGC
            ENDIF
            IF (.NOT.CONC .AND. INTERM) THEN
               IF (WDONLY) THEN
C                 Plume is above mixing height so set CONC = 0.0
                  VCOMP = 0.0
               ELSE
C                 For Concentration, Complete Vertical Term is Needed for
C                 Each Particle Size Calculated at ZFLAG ---   CALL VERT
                  CALL VERT(HECMP1,SZCMP1,A0,ZFLAG,VCOMP)
C                 Calculate Concentration for Intermediate Terrain Check
                  VCOMP = CORR*ADJ*PCORZRGC*VCOMP/SZCMP1
                  IF (WETSCIM) VCOMPD = VCOMP / WQCORGC
               END IF
            END IF

         ELSE
C           Determine Deposition Correction Factors for Particles
            IF (LDPART .OR. LWPART) THEN
               CALL PDEPC (WDONLY)
            ENDIF
C           Calculate the Vertical Term, V for particles
            DO ITYP = 1, NUMTYP
               V(ITYP) = 0.
               IF (WETSCIM) VDRY(ITYP) = 0.
            END DO
            VCOMP = 0.
            IF (WETSCIM) VCOMPD = 0.0
            DO J = 1, NPD
               ITYP = 0
C              Settling may alter SZ for the Jth particle plume
               SZADJ = SZCMP1*SZCORC(J)
               A0 = -0.5/(SZADJ*SZADJ)
C              Calculate Plume Tilt Due to Settling, HV
               HV = (X/US) * VGRAV(J)
C              Calculate Settled Plume Height, HESETL
               HESETL = HECOMP - HV
C              Restrict settled height to be positive, so that the plume
C              does not settle below the surface -- this is the limit of
C              the tilted plume technique.
               HESETL = MAX(0.0,HESETL)
C              Calculate Adjusted Plume Height and Attenuation Factor
C              for This Particle Category
               CALL CTERAD(HESETL,ZELEV,HECMP1,CORRJ)
C              Adjust Jth contribution by mass fraction and source
C              depletion
               ADJ = PHI(J) * DQCORC(J) * WQCORC(J)
               IF (CONC) THEN
C                 Concentration
                  ITYP = ITYP + 1
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set CONC = 0.0
                     V(ITYP) = 0.0
                  ELSE
C                    For Concentration, Complete Vertical Term is Needed for
C                    Each Particle Size Calculated at ZFLAG ---   CALL VERT
                     CALL VERT(HECMP1,SZADJ,A0,ZFLAG,VJ)
                     VTMP = CORRJ*ADJ*PCORZRC(J)*VJ/SZADJ
                     V(ITYP) = V(ITYP) + VTMP
                     IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP)+VTMP/WQCORC(J)
                  END IF
               END IF
               IF (DEPOS .OR. DDEP) THEN
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set DRYFLUX = 0.0
                     DRYFLUX = 0.0
                  ELSE
C                    For Dry Deposition, Complete Vertical Term is Needed for
C                    Each Particle Size Calculated at ZRDEP ---   CALL VERT
                     CALL VERT(HECMP1,SZADJ,A0,ZRDEP,VJ)
C                    Calculate Dry Flux VJ/SZ
                     DRYFLUX = CORRJ*ADJ*PCORZDC(J)*VDEP(J)*VJ/SZADJ
                  END IF
               END IF
               IF (DEPOS .OR. WDEP) THEN
C                 Calculate Wet Flux VJ/SZ --
C                 For Wet Flux, Vertical Term is Integral of EXP terms
C                 Over All z, so VJ/SZ=SQRT(2PI)
                  WETFLUX = CORRJ*ADJ*PSCVRT(J)*SRT2PI
               ENDIF
               IF (DEPOS) THEN
C                 Wet & Dry fluxes of particles are summed
                  ITYP = ITYP + 1
                  VTMP = DRYFLUX + WETFLUX
                  V(ITYP) = V(ITYP) + VTMP
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP) + VTMP/WQCORC(J)
               END IF
               IF (DDEP) THEN
C                 Dry flux of particles
                  ITYP = ITYP + 1
                  V(ITYP) = V(ITYP) + DRYFLUX
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP)+DRYFLUX/WQCORC(J)
               END IF
               IF (WDEP) THEN
C                 Wet flux of particles
                  ITYP = ITYP + 1
                  V(ITYP) = V(ITYP) + WETFLUX
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP)+WETFLUX/WQCORC(J)
               ENDIF
               IF (.NOT.CONC .AND. INTERM) THEN
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set CONC = 0.0
                     VCOMP = 0.0
                  ELSE
C                    For Concentration, Complete Vertical Term is Needed for
C                    Each Particle Size Calculated at ZFLAG ---   CALL VERT
                     CALL VERT(HECMP1,SZADJ,A0,ZFLAG,VJ)
C                    Calculate Concentration for Intermediate Terrain Check
                     VTMP  = CORRJ*ADJ*PCORZRC(J)*VJ/SZADJ
                     VCOMP = VCOMP + VTMP
                     VCOMPD= VCOMPD+ VTMP/WQCORC(J)
                  END IF
               END IF
            ENDDO
         END IF

C        Calculate the Decay Term, D                        ---   CALL DECAY
         CALL DECAY (X)

         DO ITYP = 1, NUMTYP
C           Calculate HRVAL for Sector Average in Complex Terrain
            HRVAL(ITYP) = (QTK*EMIFAC(ITYP)*D*V(ITYP)) /
     &                    (SRT2PI*DISTR*DELTHP*US)
            IF (WETSCIM) THEN
C              Repeat the above calculations for HRVALD & HRVALJD arrays
               HRVALD(ITYP) = (QTK*EMIFAC(ITYP)*D*VDRY(ITYP)) /
     &                        (SRT2PI*DISTR*DELTHP*US)
            ENDIF
         END DO

         IF (.NOT.CONC .AND. INTERM) THEN
C           Calculate Concentration for Sector Average in Complex Terrain
            COMCON = (QTK*EMICON*D*VCOMP) /
     &               (SRT2PI*DISTR*DELTHP*US)
         ELSE IF (CONC .AND. INTERM) THEN
            COMCON = HRVAL(1)
         END IF

      ELSE
C        Receptor is outside of sector or Plume is > 400m Below Receptor
         DO ITYP = 1, NUMTYP
            HRVAL(ITYP) = 0.0
            IF (WETSCIM) HRVALD(ITYP) = 0.0
         END DO
         COMCON = 0.0
      END IF

      IF (DEBUG) THEN
         WCMP1 = DELTHP * DISTR
         WRITE(IOUNIT,*) 'PCOMPL ----------------------------------'
         WRITE(IOUNIT,*) 'Hour, Receptor     =',IHOUR,IREC
         WRITE(IOUNIT,*) '  '
         WRITE(IOUNIT,*) 'QTK, D             =',QTK,D
         WRITE(IOUNIT,*) 'CORRJ, WCMP1, US   =',CORRJ,WCMP1,US
         WRITE(IOUNIT,*) 'PCOMPL ----------------------------------'
      END IF

      IF (DEBUG) THEN
C        Print Out Debugging Information                    ---   CALL DEBOUT
CRWB         CALL DEBOUT
      END IF

      RETURN
      END


      SUBROUTINE ASIMPL(XARG,RCZ,RCZD)
C***********************************************************************
C               ASIMPL Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Hourly Concentration or Deposition
C                 value for AREA Sources Using Numerical
C                 Integration Algorithm for Simple Terrain
C
C                 (Replaces ACHI and ADEP)
C
C           NOTE: Particle settling is treated as a "tilted plume"
C                 until the centerline reaches the surface.  Thereafter
C                 the centroid height of the plume continues to be
C                 modified by gravity.  This process is simulated by
C                 altering the sigma-z for each particle-size.  Hence,
C                 sigma-z is now a function of particle-size.
C
C        PROGRAMMER: D. Strimaitis, SRC
C
C        DATE:    December 15, 1993
C
C        MODIFIED:   To correct problem if CONC DDEP and WDEP are selected
C                    for area sources (DDEP results were repeated for WDEP
C                    under previous version).  Also corrects test for
C                    WDONLY (wet deposition only) flag.
C                    R. W. Brode, PES, Inc. - 12/29/97
C
C        INPUTS:  Downwind Distance (m), XARG
C
C        OUTPUTS: Relative Vertical Component of Concentration or Deposition
C                 for A Unit Of Source/Receptor Combination, RCZ
C
C        CALLED FROM:   PLUMEF
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: J
      REAL    :: RCZ, XARG, A0, SZADJ, HV, ADJ, VJ, DRYFLUX, WETFLUX
      REAL    :: RCZD, VTMP
      LOGICAL SCONC, SDEPOS, SDDEP, SWDEP, WDONLY

C     Variable Initializations
      MODNAM = 'ASIMPL'
      WDONLY = .FALSE.

C     Determine appropriate output type for this ITYP, assign output type
C     logicals to local variables, and set others to .FALSE.
      IF (OUTTYP(ITYP) .EQ. 'CONC') THEN
         SCONC  = .TRUE.
         SDEPOS = .FALSE.
         SDDEP  = .FALSE.
         SWDEP  = .FALSE.
      ELSE IF (OUTTYP(ITYP) .EQ. 'DEPOS') THEN
         SCONC  = .FALSE.
         SDEPOS = .TRUE.
         SDDEP  = .FALSE.
         SWDEP  = .FALSE.
      ELSE IF (OUTTYP(ITYP) .EQ. 'DDEP') THEN
         SCONC  = .FALSE.
         SDEPOS = .FALSE.
         SDDEP  = .TRUE.
         SWDEP  = .FALSE.
      ELSE IF (OUTTYP(ITYP) .EQ. 'WDEP') THEN
         SCONC  = .FALSE.
         SDEPOS = .FALSE.
         SDDEP  = .FALSE.
         SWDEP  = .TRUE.
      END IF

      IF ((UNSTAB .OR. NEUTRL) .AND. HEFLAT.GT.ZI) THEN
C        Plume is above mixing height, ZI
         IF (SDEPOS .OR. SWDEP) THEN
C           Set WDONLY flag for Wet Deposition Only
            WDONLY = .TRUE.
         ELSE
            V(ITYP) = 0.0
            RCZ     = 0.0
            IF (WETSCIM) VDRY(ITYP) = 0.0
            IF (WETSCIM) RCZD = 0.0
            RETURN
         END IF
      END IF

      RCZ  = 0.0
      RCZD = 0.0
      IF (XARG .GE. 1.0) THEN
         IF (NPD .EQ. 0) THEN
C           Determine Deposition Correction Factors for Gases
            IF (.NOT.ARDPLETE .AND. (LDGAS .OR. LWGAS) ) THEN
               CALL PDEPG (XARG, WDONLY)
            ENDIF
            V(ITYP) = 0.0
            VDRY(ITYP) = 0.0
            A0  = -0.5/(SZ*SZ)
            ADJ = DQCORG * WQCORG
C           Calculate the Vertical Term, V, for gases
            IF (SCONC) THEN
               IF (WDONLY) THEN
C                 Plume is above mixing height so set CONC = 0.0
                  V(ITYP) = 0.0
                  IF (WETSCIM) VDRY(ITYP) = 0.0
               ELSE
C                 Calculate Concentration Form of V         ---   CALL VERT
                  CALL VERT(HE,SZ,A0,ZFLAG,V(ITYP))
                  V(ITYP) = ADJ*PCORZRG*V(ITYP)/SZ
                  IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
               END IF
            END IF
            IF (SDEPOS .OR. SDDEP) THEN
               IF (WDONLY) THEN
C                 Plume is above mixing height so set DDEP = 0.0
                  DRYFLUX = 0.0
               ELSE
C                 For Dry Deposition Complete Vertical Term is Needed
C                 Calculated at ZRDEP                       ---   CALL VERT
                  CALL VERT(HE,SZ,A0,ZRDEP,VTMP)
                  DRYFLUX = ADJ*PCORZDG*VDEPG*VTMP/SZ
               END IF
            END IF
            IF (SDEPOS .OR. SWDEP) THEN
C              Calculate Wet Flux
C              For Wet Flux, Vertical Term is Integral of EXP terms
C              Over All z, so VJ/SZ=SQRT(2PI)
               WETFLUX = ADJ*GSCVRT*SRT2PI
            ENDIF
            IF (SDEPOS) THEN
C              Wet & Dry fluxes of particles are summed
               V(ITYP) = DRYFLUX + WETFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
            END IF
            IF (SDDEP) THEN
C              Dry flux of particles
               V(ITYP) = DRYFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
            END IF
            IF (SWDEP) THEN
C              Wet flux of particles
               V(ITYP) = WETFLUX
               IF (WETSCIM) VDRY(ITYP) = V(ITYP) / WQCORG
            ENDIF

         ELSE
C           Determine Deposition Correction Factors for Particles
            IF (.NOT.ARDPLETE .AND. (LDPART .OR. LWPART) ) THEN
               CALL PDEP (XARG, WDONLY)
            ENDIF

C           Calculate the Vertical Term, V for particles
            V(ITYP) = 0.0
            IF (WETSCIM) VDRY(ITYP) = 0.0
            DO J = 1, NPD
C              Settling may alter SZ for the Jth particle plume
               SZADJ = SZ*SZCOR(J)
               A0 = -0.5/(SZADJ*SZADJ)
C              Calculate Plume Tilt Due to Settling, HV
               HV = (XARG/US) * VGRAV(J)
C              Calculate Settled Plume Height, HESETL
               HESETL = HE - HV
C              Restrict settled height to be positive, so that the plume
C              does not settle below the surface -- this is the limit of
C              the tilted plume technique.
               HESETL = MAX(0.0,HESETL)
C              Adjust Jth contribution by mass fraction and source
C              depletion
               ADJ = PHI(J) * DQCOR(J) * WQCOR(J)
               IF (SCONC) THEN
C                 Concentration
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set CONC = 0.0
                     V(ITYP) = 0.0
                  ELSE
C                    For Concentration, Complete Vertical Term is Needed for
C                    Each Particle Size Calulated at ZFLAG     ---   CALL VERT
                     CALL VERT(HESETL,SZADJ,A0,ZFLAG,VJ)
                     VTMP    = ADJ*PCORZR(J)*VJ/SZADJ
                     V(ITYP) = V(ITYP) + VTMP
                     IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP) +VTMP/WQCOR(J)
                  END IF
               ELSE IF (SDEPOS) THEN
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set DRYFLUX = 0.0
                     DRYFLUX = 0.0
                  ELSE
C                    For Dry Deposition, Complete Vertical Term is Needed for
C                    Each Particle Size Calulated at ZRDEP     ---   CALL VERT
                     CALL VERT(HESETL,SZADJ,A0,ZRDEP,VJ)
C                    Calculate Dry Flux VJ/SZ
                     DRYFLUX = ADJ*PCORZD(J)*VDEP(J)*VJ/SZADJ
                  END IF
C                 Calculate Wet Flux VJ/SZ --
C                 For Wet Flux, Vertical Term is Integral of EXP terms
C                 Over All z, so VJ/SZ=SQRT(2PI)
                  WETFLUX = ADJ*PSCVRT(J)*SRT2PI
C                 Wet & Dry fluxes of particles are summed
                  VTMP = DRYFLUX + WETFLUX
                  V(ITYP) = V(ITYP) + VTMP
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP) + VTMP / WQCOR(J)
               ELSE IF (SDDEP) THEN
                  IF (WDONLY) THEN
C                    Plume is above mixing height so set DDEP = 0.0
                     V(ITYP) = 0.0
                     VDRY(ITYP) = 0.0
                  ELSE
C                    For Dry Deposition, Complete Vertical Term is Needed for
C                    Each Particle Size Calulated at ZRDEP     ---   CALL VERT
                     CALL VERT(HESETL,SZADJ,A0,ZRDEP,VJ)
C                    Calculate Dry Flux VJ/SZ
                     DRYFLUX = ADJ*PCORZD(J)*VDEP(J)*VJ/SZADJ
C                    Dry flux of particles
                     V(ITYP) = V(ITYP) + DRYFLUX
                     IF (WETSCIM) VDRY(ITYP)=VDRY(ITYP)+DRYFLUX/WQCOR(J)
                  END IF
               ELSE IF (SWDEP) THEN
C                 Calculate Wet Flux VJ/SZ --
C                 For Wet Flux, Vertical Term is Integral of EXP terms
C                 Over All z, so VJ/SZ=SQRT(2PI)
                  WETFLUX = ADJ*PSCVRT(J)*SRT2PI
C                 Wet flux of particles
                  V(ITYP) = V(ITYP) + WETFLUX
                  IF (WETSCIM) VDRY(ITYP) = VDRY(ITYP)+WETFLUX/WQCOR(J)
               ENDIF
            END DO
         END IF

C        Calculate the Decay Term, D                        ---   CALL DECAY
         CALL DECAY (XARG)

C        Complete TERM (SZ already in denomenator of V)
         RCZ = (D*V(ITYP))/(SRT2PI)
         IF (WETSCIM) RCZD = (D*VDRY(ITYP))/(SRT2PI)

      END IF

      RETURN
      END


      SUBROUTINE DEBOUT
C***********************************************************************
C                 DEBOUT Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Outputs Debugging Information: Sigmas, Plume Heights,
C                 etc., for Each Calculation
C
C        PROGRAMMER: Roger Brode, Jeff Wang
C
C        DATE:    March 2, 1992
C
C        INPUTS:  Downwind Distance
C                 Crosswind Distance
C                 Plume Height
C                 Stack Top Wind Speed
C                 Lateral Dispersion Parameter
C                 Vertical Dispersion Parameter
C                 Stability Class
C                 Mixing Height
C                 Receptor Height Above Ground
C                 Emission Rate and Units Scaling Factor
C                 Source Parameter Arrays
C
C        OUTPUTS: Debug Outputs
C
C        CALLED FROM:   PSIMPL, AREAIN
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE

C     Variable Initializations
      MODNAM = 'DEBOUT'

      WRITE(IOUNIT,119) JDAY, IHOUR, KURDAT, ISRC, IREC
 119  FORMAT(/1X,'JDAY= ',I3,'  IHOUR= ',I5,'  KURDAT= ',I10,
     &       '  ISRC= ',I3,'  IREC= ',I3)

      IF (SRCTYP(ISRC) .EQ. 'POINT') THEN
         WRITE(IOUNIT,209) QS, HS, TS, VS, DS, DSBH, DSBW, US
 209     FORMAT(1X,'QS= ',F8.2,' HS= ',F8.2,' TS= ',F8.2,' VS= ',
     &       F8.2,' DS= ',F8.2,' DSBH= ',F8.2,' DSBW= ',F8.2,
     &       ' US= ',F8.5)
         WRITE(IOUNIT,219) FB, FM
 219     FORMAT(1X,' FB= ',F11.5,' FM= ',F11.5)
      ELSE IF (SRCTYP(ISRC) .EQ. 'VOLUME') THEN
         WRITE(IOUNIT,229) QS, HS, SYINIT, SZINIT, US
 229     FORMAT(1X,'QS= ',F8.2,'  HS= ',F8.2,'  SYINIT= ',F8.2,
     &          '  SZINIT= ',F8.2,'  US= ',F8.5)
      ELSE IF (SRCTYP(ISRC) .EQ. 'AREA') THEN

         WRITE(IOUNIT,239) QS, HS, XINIT, US, E, SZINIT
 239     FORMAT(1X,'QS= ',F8.2,'  HS= ',F8.2,'  XINIT= ',F8.2,
     &          '  US= ',F8.5,'  E= ',G14.8,
     &          '  SZINIT= ',F8.2)

      END IF

      WRITE(IOUNIT,319) X, Y, XY, XZ, SY, SZ
 319  FORMAT(1X,'X= ',F12.4,'  Y= ',F12.4,'  XY= ',F11.5,'  XZ= ',
     &       F11.5,'  SY= ',F11.5,'  SZ= ',F12.5)
      IF (CONC) THEN
         WRITE(IOUNIT,409) HE, HEMWAK, HEFLAT, KST, TA, ZI, V(1), D
 409     FORMAT(1X,'HE= ',F11.5,'  HEMWAK= ',F11.5,'  HEFLAT= ',F11.5,
     &       '  KST= ',I2,'  TA= ',F6.1,'  ZI= ',F8.2,'  V= ',E12.6,
     &       '  D= ',E12.6)
      ELSE
         WRITE(IOUNIT,419) HE, HEMWAK, HEFLAT, KST, TA, ZI, V(1), D
 419     FORMAT(1X,'HE= ',F11.5,'  HEMWAK= ',F11.5,'  HEFLAT= ',F11.5,
     &       '  KST= ',I2,'  TA= ',F6.1,'  ZI= ',F8.2,'  V= ',E12.6,
     &       '  D= ',E12.6)
      END IF
      WRITE(IOUNIT,519) ZLB, RINIT, ZLY, DA, WAKE, WAKESS
 519  FORMAT(1X,'ZLB=',F11.5,'  RINIT= ',F9.4,'  ZLY= ',
     &       F9.4,'  DA= ',F8.6,'  WAKE= ',L3,'  WAKESS=',L3)
      WRITE(IOUNIT,619) QTK, XF, XFB, XFM, DHF, DHP
 619  FORMAT(1X,'QTK= ',E12.5,'  XF= ',F9.2,
     &       '  XFB= ',F9.2,'  XFM= ',F9.2,'  DHF= ',F9.2,
     &       '  DHP= ',F9.2)

      WRITE(IOUNIT,719) HRVAL(1)
 719  FORMAT(1X,'*** HRVAL= ',G16.8,' ***')

      RETURN
      END

c----------------------------------------------------------------------
      subroutine vdp
c----------------------------------------------------------------------
c
c --- ISC2ST     Version:  1.0     Level:  930215                   VDP
c                J. Scire, SRC
c
c --- MODIFIED   December 29, 1997
c                Removed assignment of deposition reference height, ZRDEP,
c                which is now assigned a value of 1.0m in SUB. VDP1.
c                R. W. Brode, PES, Inc.
c
c --- MODIFIED   May 26, 1995
c                Modified atmospheric resistance term, ra, based on
c                D. Byun and R. Dennis, Atmos. Environ., Vol. 29, No. 1
c                R. W. Brode, PES, Inc.
c
c --- MODIFIED   March 9, 1994
c                Changed procedure for estimating the deposition layer
c                resistance.
c                D.T. Bailey, USEPA
c
c --- PURPOSE:  Compute particle deposition velocities for each size
c               category of a size distribution.
c
c --- INPUTS:
c     Common block /METVAR/ variables:
c               Z0M - real       - Surface roughness length (m)
c             USTAR - real       - Friction velocity (m/s)
c                EL - real       - Monin-Obukhov length (m)
c     Common block /CALCS3/ variables:
c               NPD - integer    - Number of particle size categories
c             PDIAM - real array - Mean diameter (microns) of each
c                                  particle size category
c               PHI - real array - Mass fraction in each size category
c             PDENS - real       - Particle density (g/cm**3)
c                SC - real array - Schmidt number
c             VGRAV - real array - Gravitational settling velocity (m/s)
c             TSTOP - real array - Stopping time (s)
c     Common block /SOURC4/ variables:
c            VAIRMS - real       - Viscosity of air (m**2/s)
c             ZRDEP - real       - Reference height (m)
c            VDPHOR - real       - Phoretic effects term (m/s)
c
c --- OUTPUT:
c     Common block /CALCS3/ variables:
c              VDEP - real array - Deposition velocity (m/s) for each
c                                  particle size category
c
c --- VDP called by:  PCALC, VCALC, ACALC
c --- VDP calls:      none
c----------------------------------------------------------------------
c
      USE MAIN1
      IMPLICIT NONE

      SAVE
      INTEGER :: I, IO6, N
      real :: USTARR, ELL, ELABS, PSIH, RA, A1, B1, T1, ST, XINERT,
     &        SCHMIDT, rd(npdmax), RDG, RG, B, FR, RS, RF, RC
c
      io6=iounit
c
c ***
      if(DEBUG)then
         WRITE(IO6,*)'IHOUR  = ',IHOUR
         WRITE(IO6,*)'ISTAHR = ',ISTAHR
         WRITE(IO6,*)'IENDHR = ',IENDHR
         WRITE(IO6,*)'IEVENT = ',IEVENT
         WRITE(IO6,*)'USTAR(IHOUR) = ',AUSTAR(IHOUR)
         write(io6,*)
         write(io6,*)'SUBR. VDP -- Inputs'
         write(io6,*)'USTAR (m/s)     = ',ustar
         write(io6,*)'MONIN-EL (m)    = ',el
         write(io6,*)'Z0M (m)         = ',z0m
         write(io6,*)'VDPHOR (m/s)    = ',vdphor
         write(io6,*)'NPD             = ',npd
         write(io6,*)'PDIAM (um)      = ',(pdiam(n),n=1,npd)
         write(io6,*)'FRACT           = ',(phi(n),n=1,npd)
         write(io6,*)'PDENS (g/cm**3) = ',(pdens(n),n=1,npd)
         write(io6,*)'SC              = ',(sc(n),n=1,npd)
         write(io6,*)'VGRAV (m/s)     = ',(vgrav(n),n=1,npd)
         write(io6,*)'TSTOP (s)       = ',(tstop(n),n=1,npd)
         write(io6,*)'VAIRMS (m**2/s) = ',vairms
         write(io6,*)'ZRDEP (m)       = ',zrdep
         write(io6,*)'VDPHOR (m/s)    = ',vdphor
      endif
c ***
c
c --- Use minimum value of USTAR to avoid numerical problems
c --- when USTAR near zero
      ustarr=MAX(ustar,1.e-9)
c
c --- Minimum absolute value of Monin-Obukhov length is 1.0 m
      if(el.GE.0.0)then
c ---    stable
         ell=MAX(el,1.0)
      else
c ---    unstable
         ell=MIN(el,-1.0)
      endif
c
c --- Calculate atmospheric resistance (s/m)
      elabs=ABS(ell)
      if (ell .gt. 0.0) then
c ---    Stable
c ---    VK is the von Karman constant, set as parameter in MAIN1
         psih = 4.7*zrdep/ell
         ra = (1.0/(vk*ustarr)) * (LOG(zrdep/z0m) + psih)

      else
c ---    Unstable
         a1 = 16.*zrdep/elabs
         b1 = 16.*z0m/elabs
         ra = (1.0/(vk*ustarr)) * (1.0*LOG(
     &        ((2.+a1)-2.*SQRT(1.+a1)) * ((2.+b1)+2.*SQRT(1.+b1)) /
     &        (a1*b1) ))
      endif
c
c ***
      if(DEBUG)then
         write(io6,*)
         write(io6,*)'USTARR (m/s)    = ',ustarr
         write(io6,*)'ELL (m)         = ',ell
         write(io6,*)'PSIH            = ',psih
      endif
c ***
c
      if (npd .eq. 0 .and. LUSERVD) then
c
c ---    GAS DEPOSITION with User-specified Deposition Velocity
c
         vdepg = uservd
c ***
         if(DEBUG)then
            write(io6,*)
            write(io6,*)'User-specified deposition velocity:'
            write(io6,*)'VDEPG (m/s) = ',vdepg
         endif
c ***

      else if (npd .eq. 0) then
c
c ---    GAS DEPOSITION
c
c ---    Compute the deposition layer resistance for gases, RDG
c ---    (RD1 is d1*SC**d2/vk -- computed in setup routine)
         RDG=RD1(isrc)/ustarr
c
c ---    Compute resistance directly to ground or water, RG
c ---    Water is assumed if LAI = 0.0
         if (XLAI .EQ. 0.0) then
c
c ---       Water cell (RGW1 computed in setup routine as
c ---                   RGW1 = HENRY/(ALPHAS * D3)
            RG=RGW1(isrc)/ustarr
         else
c
c ---       Land cell (RG computed in setup routine)
            RG=RGG(isrc)
         endif
c
c ---    Stomatal pore resistance (RS)
         if (UNSTRESSED) then
c
c ---       Vegetation is active & unstressed (IVEG=1 in CALPUFF)
c           (B = stomatal pore opening (m), BMIN = minimum stomatal
c           opening, BMAX = maximum stomatal opening, fr is the approx.
c           fraction of peak short-wave solar radiation available for a
c           particular hour)
c
c ---       Temperature effects -- If T > 35 deg. C, stomata fully
c ---       open to allow evaporative cooling -- but only if unstressed --
c ---       If T < 10 deg. C, stomata closed due to decreased metabolic
c ---       activity)
            if(ta.gt.308.)then
c ---          T > 35 deg. C
               B=BMAX
            else if(ta.lt.283.)then
c ---          T < 10 deg. C
               B=BMIN
            else
               fr=qsw/qswmax
               fr=MAX(0.0,fr)
               fr=MIN(1.0,fr)
               B=BMAX*fr+BMIN*(1.-fr)
            endif
c
            RS=pconst/(B*pdiff(isrc))
         else if (STRESSED) then
c
c ---       Vegetation is active and stressed (IVEG=2 in CALPUFF)
c           (Stomatal opening is at its minimum size)
            RS=pconst/(BMIN*pdiff(isrc))
         else if (INACTIVE) then
c
c ---       Vegetation is inactive (IVEG=3)
            RS=9.9e9
         endif
c
c ---    Internal foliage resistance (RF)
c        (RM is the mesophyll resistance)
         RF=RS+RM(isrc)
c
c ---    Compute canopy resistance
         RC=1.0/(XLAI/RF+XLAI/RCUT(isrc)+1.0/RG)
c
c ---    Deposition velocity is the inverse of the sum of the
c ---    atmospheric, deposition layer, and canopy resistances
         vdepg = 1.0/(ra+rdg+rc)

c ***
         if(DEBUG)then
            write(io6,*)
            write(io6,*)'RA (s/m)    = ',ra
            write(io6,*)'RDG (s/m)   = ',rdg
            write(io6,*)'RC (s/m)    = ',rc
            write(io6,*)'VDEPG (m/s) = ',vdepg
         endif
c ***

      else

c
c ---    PARTICLE DEPOSITION
c
         t1=ustarr*ustarr/vairms
c
c ---    LOOP OVER SIZE INTERVALS
         do i=1,npd
c
            st=tstop(i)*t1
c
c ---       Compute inertial impaction term
            xinert=10**(-3./st)
c
c ---       Adjust (raise) the Schmidt Number to the 2/3rd's power.
            Schmidt = sc(i) ** (-.667)                                       DTB94068
c
c ---       Compute the deposition layer resistance (s/m)
            rd(i)=1.0 / (ustarr * (Schmidt + xinert))                        DTB94068
c
c ---       Deposition velocity for this current interval
            vdep(i)=1.0/(ra+rd(i)+ra*rd(i)*vgrav(i))+vgrav(i)+vdphor

         end do
c ***
         if(DEBUG)then
            write(io6,*)
            write(io6,*)'RA (s/m)    = ',ra
            write(io6,*)'RD (s/m)    = ',(rd(n),n=1,npd)
            write(io6,*)'VDEP (m/s)  = ',(vdep(n),n=1,npd)
         endif
c ***
c
      endif
c
      return
      end

c-----------------------------------------------------------------------
      subroutine setszmn
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           SETSZMN
c               D. Strimaitis, SRC
c
c PURPOSE:     SETSZMN determines the value of sigma-z at which the rate
c              of growth in sigma-z equals the rate at which the settling
c              velocity acts to reduce the height of the center-of-mass.
c              A default minimum of 2*zd, where zd is the near-surface
c              height at which the deposition flux is evaluated, is
c              returned if there is no balance-point.
c
c ARGUMENTS:  (MAIN1)
c    PASSED:  kst       stability class (A=1, F=6)                   [i]
c             zrdep     reference height for deposition flux  (m)    [r]
c             vs        settling velocity  (m/s)                     [r]
c             us        plume advection wind speed (m/s)             [r]
c             urban     logical for URBAN/RURAL dispersion params    [l]
c             npd       number of particle size categories           [i]
c
c  RETURNED:  szmin     Minimum value of sigma-z (m)                 [r]
c
c CALLING ROUTINES:   PCALC, VCALC, ACALC
c
c EXTERNAL ROUTINES:  GCUBIC
c-----------------------------------------------------------------------
      USE MAIN1
      IMPLICIT NONE

      SAVE
      INTEGER :: I, J
      REAL    :: XMIN, A, B, C, A1, A2, A3, ABY2CSQ

      real root(3),car(6),cau(6),cbr(6),cbu(6)

      data car/.2,.12,.08,.06,.03,.016/
      data cbr/0.,0.,.0002,.0015,.0003,.0003/
      data cau/.24,.24,.2,.14,.08,.08/
      data cbu/.001,.001,0.,.0003,.0015,.0015/


c --- Loop over particle sizes
      do i=1,npd
         xmin=0.0
         szmin(i)=2.*zrdep
         c=rtpiby2*vgrav(i)/us

c ---    Urban section
         if(URBAN) then
            a=cau(kst)
            b=cbu(kst)
            if(kst .GE. 4) then
               if(a .GT. 20.*c) then
                  szmin(i)=a*a/(2.*b*c)
               elseif(a .GT. c) then
c ---             Solve cubic for y=bx, then report x       ---  call GCUBIC
                  aby2csq=(a/(2.*c))**2
                  a1=(3.-aby2csq)
                  a2=(3.-4.*aby2csq)
                  a3=(1.-4.*aby2csq)
                  call GCUBIC(a1,a2,a3,root)
c ---             There should be ONE real root
                  if(root(2) .NE. 0. .OR. root(3) .NE. 0.) then
                     write(*,*) 'SETSZMN: Potential error!!! '
                     write(*,*) 'More than 1 root ----'
                     write(*,*) 'xb= ',(root(j),j=1,3)
                  endif
                  xmin=root(1)/b
                  szmin(i)=a*xmin/SQRT(1.+b*xmin)
               endif
            endif

c ---    Rural section
         else
            a=car(kst)
            b=cbr(kst)
            if(kst .EQ. 3 .OR. kst .EQ. 4) then
               if(a .GT. 20.*c) then
                  szmin(i)=a*a/(2.*b*c)
               elseif(a .GT. c) then
c ---             Solve cubic for y=bx, then report x       ---  call GCUBIC
                  aby2csq=(a/(2.*c))**2
                  a1=(3.-aby2csq)
                  a2=(3.-4.*aby2csq)
                  a3=(1.-4.*aby2csq)
                  call GCUBIC(a1,a2,a3,root)
c ---             There should be ONE real root
                  if(root(2) .NE. 0. .OR. root(3) .NE. 0.) then
                     write(*,*) 'Potential error!!! More than 1 root'
                     write(*,*) 'xb= ',(root(j),j=1,3)
                  endif
                  xmin=root(1)/b
                  szmin(i)=a*xmin/SQRT(1.+b*xmin)
               endif
            elseif(kst .GT. 4) then
               if(a .GT. c) then
                  xmin=(SQRT(a/c)-1.)/b
                  szmin(i)=a*xmin/(1+b*xmin)
               endif
            endif
         endif

      enddo

      return
      end

c-----------------------------------------------------------------------
      subroutine gcubic(a1,a2,a3,root)
c-----------------------------------------------------------------------
c
c --- ISCST2    Version: 1.0            Level: 931215           GCUBIC
c               D. Strimaitis, SRC
c
c PURPOSE:     Program solves the general cubic equation of the form:
c                  0 = x**3 + (a1)x**2 + (a2)x + (a3)
c              for the real roots
c              (Numerical Recipes, Press et al., 1986)
c
c ARGUMENTS:
c    PASSED:  a1,a2,a3  constants for terms as described above       [r]
c
c  RETURNED:  root      root(s) of equation                          [r]
c
c CALLING ROUTINES:   (utility routine)
c
c EXTERNAL ROUTINES:  none
c-----------------------------------------------------------------------

      IMPLICIT NONE

      real :: A1, A2, A3, THIRD, A1SQ, A1CUBE, A1BY3, Q, R, QCUBE,
     &        RSQ, SQRTQ2, THETA, ARG, root(3)

      REAL, PARAMETER :: twopi=6.2831853, fourpi=12.566371

      third=1./3.
      a1sq=a1*a1
      a1cube=a1*a1sq
      a1by3=a1*third

      q=(a1sq-3.*a2)/9.
      r=(2.*a1cube-9.*a1*a2+27.*a3)/54.

      qcube=q*q*q
      rsq=r*r

      if(qcube .GE. rsq) then
c ---    THREE real roots
         sqrtq2=SQRT(q)*2.
         theta=ACOS(r/SQRT(qcube))
         root(1)=-sqrtq2*COS(theta/3.)-a1by3
         root(2)=-sqrtq2*COS((theta+twopi)/3.)-a1by3
         root(3)=-sqrtq2*COS((theta+fourpi)/3.)-a1by3
      else
c ---    ONE real root
         arg=(SQRT(rsq-qcube)+ABS(r))**third
         root(1)=-SIGN(1.0,r)*(arg+q/arg)-a1by3
         root(2)=0.
         root(3)=0.
      endif


      return
      end

c----------------------------------------------------------------------
      subroutine scavrat
c----------------------------------------------------------------------
c
c --- ISCST2     Version: 1.0       Level: 931108               SCAVRAT
c                D. Strimaitis, SRC
c
c --- PURPOSE:  Compute the wet SCAVenging RATio for particles, as a
c               function of particle size, and for gases
c
c --- INPUTS:
c     Common block /METVAR/ variables:
c            IPCODE - integer    - Precip. code (00-45)
c             PRATE - real       - Precip. rate (mm/hr)
c                TA - real       - Ambient Temperature (deg K)
c     Common block /CALCS3/ variables:
c               NPD - integer    - Number of particle size categories
c             PSCAV - real array - Particle scavenging coefs. for liquid
c                                  (1) and frozen (2) precip. for each
c                                  size category (1/[s-mm/hr])
c             GSCAV - real array - Gas scavenging coefs. for liquid (1)
c                                  and frozen (2) precip. (1/[s-mm/hr])
c
c --- OUTPUT:
c     Common block /CALCS3/ variables:
c            PSCVRT - real array - Scavenging ratio for particles (1/s)
c            GSCVRT - real       - Scavenging ratio for gases (1/s)
c
c --- SCAVRAT called by:  PCALC, VCALC, ACALC
c --- SCAVRAT calls:      none
c----------------------------------------------------------------------
c
c --- Include common blocks
      USE MAIN1
      IMPLICIT NONE

      SAVE
      INTEGER :: I, N, ILQ, IMISS

      data imiss/9999/

      if(DEBUG)then
         write(iounit,*)
         write(iounit,*)'SUBR. SCAVRAT -- Inputs'
         write(iounit,*)'IPCODE               = ',ipcode
         write(iounit,*)'PRATE (mm/hr)        = ',prate
         write(iounit,*)'TA (deg K)           = ',ta
         write(iounit,*)'NPD                  = ',npd
         write(iounit,*)'PSCAV(1) 1/(s-mm/hr) = ',(pscav(n,1),n=1,npd)
         write(iounit,*)'PSCAV(2) 1/(s-mm/hr) = ',(pscav(n,2),n=1,npd)
         write(iounit,*)'GSCAV(1) 1/(s-mm/hr) = ',gscav(1)
         write(iounit,*)'GSCAV(2) 1/(s-mm/hr) = ',gscav(2)
         write(iounit,*)' (1 = Liquid ; 2 = Frozen )'
         write(iounit,*)
      endif

c --- If no precipitation, no wet removal
      if(prate .EQ. 0.) then
         do i=1,npd
            pscvrt(i)=0.0
         enddo
         gscvrt=0.0
      else
c ---    Determine if precip. is liquid (ILQ=1) or frozen (ILQ=2)
         if(ipcode .EQ. imiss .OR. ipcode .EQ. 0) then
c ---       Precip. code is unavailable due to missing data or no
c           precip. at time of obs. at surface station, therefore,
c           determine precip. type based on the air temperature
c ---       Assume liquid precip. if temp. > freezing, otherwise,
c           assume frozen precip.
            if(ta .GT. 273.15) then
               ilq=1
            else
               ilq=2
            endif
         else if(ipcode .LE. 18) then
c ---       Liquid precipitation type
            ilq=1
         else
c ---       Frozen precipitation type
            ilq=2
         endif
c ---    Determine the scavenging ratios
         do i=1,npd
            pscvrt(i)=pscav(i,ilq)*prate
         enddo
         gscvrt=gscav(ilq)*prate
      endif

      if(DEBUG)then
         write(iounit,*)'SUBR. SCAVRAT -- Results'
         write(iounit,*)'GSCVRT (1/s)= ',gscvrt
         write(iounit,*)'PSCVRT (1/s)= ',(pscvrt(n),n=1,npd)
         write(iounit,*)
      endif

      return
      end


      SUBROUTINE PDEP (XARG, LWDONLY)
C***********************************************************************
C               PDEP Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Simple Terrain Deposition Adjustment
C                 Factors from DEPCOR
C
C        PROGRAMMER: R. W. Brode, PES, Inc.
C
C        DATE:    September 30, 1994
C
C        MODIFIED:   To add logical argument for Wet Deposition Only,
C                    to skip call to DEPCOR when plume is above ZI.
C                    R.W. Brode, PES, 7/17/95
C
C        INPUTS:     LWDONLY, logical specifying whether Wet Deposition
C                    Only is to be calculated for plume above ZI
C
C        OUTPUTS:
C
C
C        CALLED FROM:   PSIMPL
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I
      REAL    :: XARG
      LOGICAL LTERR, LWDONLY

C     Variable Initializations
      MODNAM = 'PDEP'

C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR = .FALSE.

C     Loop over particle sizes
      DO I=1,NPD
         DQCOR(I)  = 1.0
         PCORZR(I) = 1.0
         PCORZD(I) = 1.0
         SZCOR(I)  = 1.0
C        Initialize wet & dry source depletion factors,
C        profile correction factors, and settles sigma-z
C        factors to unity. - Done in DEPCOR
         IF (DDPLETE .AND. .NOT.LWDONLY) THEN
C           Determine factors for depletion - note that
C           plume ht adjustment for terrain is signalled
C           by a local logical - LTERR
C           Simple Terrain Model          ---   CALL DEPCOR
            CALL DEPCOR( VDEP(I),VGRAV(I),ZRDEP,ZFLAG,
     &        XARG,XZ,HEFLAT,ZI,US,XS,YS,XR,YR,
     &        RURAL,URBAN,KST,SZ,SBID,
     &        SZMIN(I),ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &        SRCTYP(ISRC),LTGRID,KURDAT,
     &        DQCOR(I),PCORZR(I),PCORZD(I),SZCOR(I),TOXICS)
         END IF
         IF (WDPLETE) THEN
C           Determine source depletion factor
C           from wet removal
C           Simple Terrain Model
            WQCOR(I) = EXP(-PSCVRT(I)*XARG/US)
         ELSE
            WQCOR(I) = 1.
         ENDIF
      END DO

      RETURN
      END


      SUBROUTINE PDEPC (LWDONLY)
C***********************************************************************
C               PDEPC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Complex Terrain Deposition Adjustment
C                 Factors from DEPCOR
C
C        PROGRAMMER: R. W. Brode, PES, Inc.
C
C        DATE:    September 30, 1994
C
C        MODIFIED:   To add logical argument for Wet Deposition Only,
C                    to skip call to DEPCOR when plume is above ZI.
C                    R.W. Brode, PES, 7/17/95
C
C        INPUTS:     LWDONLY, logical specifying whether Wet Deposition
C                    Only is to be calculated for plume above ZI
C
C        OUTPUTS:
C
C
C        CALLED FROM:   PCOMPL
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      INTEGER :: I
      LOGICAL LTERR, LWDONLY

C     Variable Initializations
      MODNAM = 'PDEPC'

C     Set LTERR to TRUE to signal complex terrain call to DEPCOR.
      LTERR = .TRUE.

C     Loop over particle sizes
      DO I=1,NPD
         DQCORC(I)  = 1.0
         PCORZRC(I) = 1.0
         PCORZDC(I) = 1.0
         SZCORC(I)  = 1.0
C        Initialize wet & dry source depletion factors,
C        profile correction factors, and settles sigma-z
C        factors to unity. - Done in DEPCOR
         IF (DDPLETE .AND. .NOT.LWDONLY) THEN
C           Determine factors for depletion - note that
C           plume ht adjustment for terrain is signalled
C           by a local logical - LTERR
C           Complex Terrain Model         ---   CALL DEPCOR
            CALL DEPCOR( VDEP(I),VGRAV(I),ZRDEP,ZFLAG,
     &        DISTR,XZCMP1,HECOMP,ZI,US,XS,YS,XR,YR,
     &        RURAL,URBAN,KST,SZCMP1,SBCMP1,
     &        SZMIN(I),ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &        SRCTYP(ISRC),LTGRID,KURDAT,
     &        DQCORC(I),PCORZRC(I),PCORZDC(I),
     &        SZCORC(I),TOXICS)
         END IF
         IF (WDPLETE) THEN
C           Determine source depletion factor
C           from wet removal
C           Complex Terrain Model - use radial distance
            WQCORC(I) = EXP(-PSCVRT(I)*DISTR/US)
         ELSE
            WQCORC(I) = 1.
         ENDIF
      END DO

      RETURN
      END

      SUBROUTINE PDEPG (XARG, LWDONLY)
C***********************************************************************
C               PDEPG Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Simple Terrain Deposition Adjustment
C                 Factors from DEPCOR for Gases
C
C        PROGRAMMER: R. W. Brode, PES, Inc.
C
C        DATE:       May 6, 1996
C
C        INPUTS:     LWDONLY, logical specifying whether Wet Deposition
C                    Only is to be calculated for plume above ZI
C
C        OUTPUTS:
C
C
C        CALLED FROM:   PSIMPL
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      REAL :: XARG
      LOGICAL LTERR, LWDONLY

C     Variable Initializations
      MODNAM = 'PDEPG'

C     Set LTERR to FALSE to signal simple terrain call to DEPCOR.
      LTERR = .FALSE.

C     Initialize source depletion factors to unity.
      DQCORG  = 1.0
      PCORZRG = 1.0
      PCORZDG = 1.0
      SZCORG  = 1.0
      WQCORG  = 1.0
C     Initialize wet & dry source depletion factors,
C     profile correction factors, and settles sigma-z
C     factors to unity. - Done in DEPCOR
      IF (DDPLETE .AND. .NOT.LWDONLY) THEN
C        Determine factors for depletion - note that
C        plume ht adjustment for terrain is signalled
C        by a local logical - LTERR
C        Simple Terrain Model          ---   CALL DEPCOR
         CALL DEPCOR( VDEPG,0.0,ZRDEP,ZFLAG,
     &     XARG,XZ,HEFLAT,ZI,US,XS,YS,XR,YR,
     &     RURAL,URBAN,KST,SZ,SBID,
     &     2.*ZRDEP,ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &     SRCTYP(ISRC),LTGRID,KURDAT,
     &     DQCORG,PCORZRG,PCORZDG,SZCORG,TOXICS)
      END IF
      IF (WDPLETE) THEN
C        Determine source depletion factor
C        from wet removal (GASES)
C        Simple Terrain Model
         WQCORG = EXP(-GSCVRT*XARG/US)
      ENDIF

      RETURN
      END

      SUBROUTINE PDEPGC (LWDONLY)
C***********************************************************************
C               PDEPGC Module of ISC2 Short Term Model - ISCST2
C
C        PURPOSE: Calculates Complex Terrain Deposition Adjustment
C                 Factors from DEPCOR for Gases
C
C        PROGRAMMER: R. W. Brode, PES, Inc.
C
C        DATE:       May 6, 1996
C
C        INPUTS:     LWDONLY, logical specifying whether Wet Deposition
C                    Only is to be calculated for plume above ZI
C
C        OUTPUTS:
C
C
C        CALLED FROM:   PCOMPL
C***********************************************************************

C     Variable Declarations
      USE MAIN1
      IMPLICIT NONE
      CHARACTER MODNAM*12

      SAVE
      LOGICAL LTERR, LWDONLY

C     Variable Initializations
      MODNAM = 'PDEPGC'

C     Set LTERR to TRUE to signal complex terrain call to DEPCOR.
      LTERR = .TRUE.

C     Initialize source depletion factors to unity.
      DQCORGC  = 1.0
      PCORZRGC = 1.0
      PCORZDGC = 1.0
      SZCORGC  = 1.0
      WQCORGC  = 1.0
C     Initialize wet & dry source depletion factors,
C     profile correction factors, and settles sigma-z
C     factors to unity. - Done in DEPCOR
      IF (DDPLETE .AND. .NOT.LWDONLY) THEN
C        Determine factors for depletion - note that
C        plume ht adjustment for terrain is signalled
C        by a local logical - LTERR
C        Simple Terrain Model          ---   CALL DEPCOR
         CALL DEPCOR( VDEPG,0.0,ZRDEP,ZFLAG,
     &     DISTR,XZCMP1,HECOMP,ZI,US,XS,YS,XR,YR,
     &     RURAL,URBAN,KST,SZCMP1,SBCMP1,
     &     2.*ZRDEP,ZELEV,ZS,LTERR,DEBUG,IOUNIT,
     &     SRCTYP(ISRC),LTGRID,KURDAT,
     &     DQCORGC,PCORZRGC,PCORZDGC,SZCORGC,TOXICS)
      END IF
      IF (WDPLETE) THEN
C        Determine source depletion factor
C        from wet removal (GASES)
C        Simple Terrain Model
         WQCORGC = EXP(-GSCVRT*DISTR/US)
      ENDIF

      RETURN
      END
